/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.KeyStroke;
import javax.swing.RootPaneContainer;

/**
 * Utils for the package.
 */
public final class ScalableLayoutUtils {

    private static final int WIDTH_FIX = 3;

    /**
     * Private constructor.
     */
    private ScalableLayoutUtils() {
        throw new UnsupportedOperationException("Cannot create instance of utility class!");
    }

    /**
     * Scales the font.
     * 
     * @param font
     *            Font to scale.
     * @param factor
     *            Factor to use.
     * 
     * @return New instance with scaled size.
     */
    public static Font scale(final Font font, final double factor) {
        if (font == null) {
            return null;
        }
        final int size = (int) Math.round(font.getSize() * factor);
        return new Font(font.getName(), font.getStyle(), size);
    }

    /**
     * Scales the rectangle.
     * 
     * @param rect
     *            Rectangle to scale.
     * @param factor
     *            Factor to use.
     * 
     * @return New instance with scaled sizes.
     */
    public static Rectangle scale(final Rectangle rect, final double factor) {
        if (rect == null) {
            return null;
        }
        final int height = (int) Math.round(rect.height * factor);
        final int width = (int) Math.round(rect.width * factor) + WIDTH_FIX;
        final int x = (int) Math.round(rect.x * factor);
        final int y = (int) Math.round(rect.y * factor);
        return new Rectangle(x, y, width, height);
    }

    /**
     * Scales the dimension.
     * 
     * @param dim
     *            Dimension to scale.
     * @param factor
     *            Factor to use.
     * 
     * @return New instance with scaled sizes.
     */
    public static Dimension scale(final Dimension dim, final double factor) {
        if (dim == null) {
            return null;
        }
        final int height = (int) Math.round(dim.height * factor);
        final int width = (int) Math.round(dim.width * factor) + WIDTH_FIX;
        return new Dimension(width, height);
    }

    /**
     * Scales a value.
     * 
     * @param value
     *            Value to scale.
     * @param factor
     *            Factor to use.
     * 
     * @return New instance with scaled sizes.
     */
    public static int scale(final int value, final double factor) {
        return (int) Math.round(value * factor);
    }

    /**
     * Scales the insets.
     * 
     * @param current
     *            Insets that will be changed (result insets).
     * @param original
     *            Insets that will be used with the factor.
     * @param factor
     *            Factor to use.
     */
    public static void scale(final Insets current, final Insets original, final double factor) {
        if ((original != null) && (current != null)) {
            current.bottom = (int) Math.round(original.bottom * factor);
            current.left = (int) Math.round(original.left * factor);
            current.right = (int) Math.round(original.right * factor);
            current.top = (int) Math.round(original.top * factor);
        }
    }

    /**
     * Install two key strokes (CTRL+ and CTRL-) for scaling the layout up and
     * down on the content pane of the root pane container.
     * 
     * @param rootPaneContainer
     *            The content pane of this root pane container will be used to
     *            install the keys on. The root pane container itself will also
     *            be scaled.
     * @param layout
     *            Layout to scale.
     * @param step
     *            Scaling steps (For example 0.1 is a good value).
     */
    @SuppressWarnings("serial")
    public static void installScaleKeys(final RootPaneContainer rootPaneContainer,
            final AbstractScalableLayout layout, final double step) {

        final Container contentPane = rootPaneContainer.getContentPane();
        if (!(contentPane instanceof JComponent)) {
            throw new IllegalArgumentException(
                    "The content pane of the root container is not of type 'JComponent'! ["
                            + contentPane.getClass().getName() + "]");
        }
        final JComponent comp = (JComponent) contentPane;

        comp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_ADD, KeyEvent.CTRL_MASK), "SCALE_UP");
        comp.getActionMap().put("SCALE_UP", new AbstractAction() {
            public void actionPerformed(final ActionEvent e) {
                final double factor = layout.getFactor() + 0.1;
                layout.setFactorRecursive(factor);
            }
        });
        comp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_SUBTRACT, KeyEvent.CTRL_MASK), "SCALE_DN");
        comp.getActionMap().put("SCALE_DN", new AbstractAction() {
            public void actionPerformed(final ActionEvent e) {
                final double factor = layout.getFactor() - 0.1;
                layout.setFactorRecursive(factor);
            }
        });
    }

    /**
     * Install the appropriate scalable layout on the container.
     * 
     * @param registry
     *            Registry to use.
     * @param container
     *            Container to replace the existing layout with a scalable
     *            layout.
     */
    public static void installScalableLayout(final ScalableLayoutRegistry registry,
            final Container container) {
        final AbstractScalableLayout layout = registry.getScalableLayout(registry, container);
        container.setLayout(layout);
    }

    /**
     * Install the appropriate scalable layout on the container and it's sub
     * containers.
     * 
     * @param registry
     *            Registry to use.
     * @param container
     *            Container to replace the existing layout with a scalable
     *            layout.
     */
    public static void installScalableLayoutRecursive(final ScalableLayoutRegistry registry,
            final Container container) {

        final Component[] comps = container.getComponents();
        for (int i = 0; i < comps.length; i++) {
            if (registry.isContainer(comps[i])) {
                installScalableLayoutRecursive(registry, (Container) comps[i]);
            }
        }
        installScalableLayout(registry, container);

    }

    /**
     * Install the appropriate scalable layout on the content pane of the
     * <code>RootPaneContainer</code> and it's sub containers. Additionally
     * <code>installScaleKeys(JComponent component,
            AbstractScalableLayout, double)</code> is called to install scaling keys
     * on the content pane.
     * 
     * @param registry
     *            Registry to use.
     * @param rootPaneContainer
     *            Content pane of this root pane container will be used. The
     *            content is expected to be a <code>JComponent</code>!
     * @param step
     *            Scaling steps (For example 0.1 is a good value).
     */
    public static void installScalableLayoutAndKeys(final ScalableLayoutRegistry registry,
            final RootPaneContainer rootPaneContainer, final double step) {

        final Container contentPane = rootPaneContainer.getContentPane();
        installScalableLayoutRecursive(registry, contentPane);
        installScaleKeys(rootPaneContainer, (AbstractScalableLayout) contentPane.getLayout(),
                step);

    }
}
