/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.appstarter;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.table.TableColumn;

import org.fuin.utils4j.Utils4J;

import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.EventList;
import ca.odell.glazedlists.swing.EventListModel;
import ca.odell.glazedlists.swing.EventTableModel;

import com.jeta.forms.components.panel.FormPanel;
import com.l2fprod.common.swing.JDirectoryChooser;

/**
 * Main panel for the "AppStarter". Configures everything before starting the
 * application.
 */
public final class AppStarterConfigPanel extends FormPanel {

    private static final long serialVersionUID = 1L;

    private final EventList<ClasspathEntry> classpathEntries;

    private final EventList<String> arguments;

    private final AppStarter appStarter;

    private File lastDir = new File(".");

    /**
     * Constructor with configuration and the "AppStarter".
     * 
     * @param config Configuration - Cannot be <code>null</code>.
     * @param appStarter Used for starting the application - Cannot be <code>null</code>.
     */
    public AppStarterConfigPanel(final AppStarterConfig config,
            final AppStarter appStarter) {
        super("AppStarterConfig.jfrm");
        Utils4J.checkNotNull("config", config);        
        Utils4J.checkNotNull("appStarter", appStarter);        
        this.appStarter = appStarter;
        this.classpathEntries = new BasicEventList<ClasspathEntry>();
        this.classpathEntries.addAll(config.getClassPath());
        this.arguments = new BasicEventList<String>();
        this.arguments.addAll(config.getArguments());

        init();

        getTextFieldMainClass().setText(config.getMainClass());
        getComboboxStartType().setSelectedItem(config.getStartType().getName());
        getLabelTitle().setText(config.getFrameTitle());

    }

    /**
     * Initializes the UI.
     */
    private void init() {

        // Common
        final int width = getPreferredSize().width + 15;
        final int height = getPreferredSize().height + 35;
        setPreferredSize(new Dimension(width, height));

        // TableClassPath
        getTableClassPath().setModel(
                new EventTableModel<ClasspathEntry>(classpathEntries,
                        new ClasspathEntryTableFormat()));
        getTableClassPath().setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        final TableColumn colType = getTableClassPath().getColumnModel()
                .getColumn(0);
        colType.setPreferredWidth(100);
        final TableColumn colValue = getTableClassPath().getColumnModel()
                .getColumn(1);
        colValue.setPreferredWidth(430);
        getTableClassPath().setSelectionMode(
                ListSelectionModel.SINGLE_SELECTION);
        getTableClassPath().getSelectionModel().addListSelectionListener(
                new ListSelectionListener() {
                    public void valueChanged(final ListSelectionEvent e) {
                        getButtonRemoveClassPath()
                                .setEnabled(
                                        (getTableClassPath()
                                                .getSelectedRowCount() > 0));
                    }
                });

        // ListArguments
        getListArguments().setModel(new EventListModel<String>(arguments));
        getListArguments()
                .setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        getListArguments().getSelectionModel().addListSelectionListener(
                new ListSelectionListener() {
                    public void valueChanged(final ListSelectionEvent e) {
                        getButtonRemoveArgument().setEnabled(
                                (getListArguments().getSelectedIndex() > -1));
                    }
                });

        // Start types
        final DefaultComboBoxModel modelStartTypes = (DefaultComboBoxModel) getComboboxStartType()
                .getModel();
        for (final StartType element : StartType.INSTANCES) {
            modelStartTypes.addElement(element.getName());
        }

        // Buttons

        getButtonAddJarFileClassPath().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                addJarFileClassPath();
            }
        });

        getButtonAddJarsDirClassPath().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                addJarsDirClassPath();
            }
        });

        getButtonAddBinDirClassPath().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                addBinDirClassPath();
            }
        });

        getButtonRemoveClassPath().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                removeClassPath();
            }
        });

        getButtonAddArgument().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                addArgument();
            }
        });

        getButtonRemoveArgument().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                removeArgument();
            }
        });

        getButtonStartApplication().addActionListener(new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                startApplication();
            }
        });

    }

    private JTable getTableClassPath() {
        return getTable("TableClassPath");
    }

    private JButton getButtonAddJarsDirClassPath() {
        return (JButton) getButton("ButtonAddJarsDirClassPath");
    }

    private JButton getButtonAddBinDirClassPath() {
        return (JButton) getButton("ButtonAddBinDirClassPath");
    }

    private JButton getButtonRemoveClassPath() {
        return (JButton) getButton("ButtonRemoveClassPath");
    }

    private JButton getButtonAddJarFileClassPath() {
        return (JButton) getButton("ButtonAddJarFileClassPath");
    }

    private JComboBox getComboboxStartType() {
        return getComboBox("ComboboxStartType");
    }

    private JTextField getTextFieldMainClass() {
        return getTextField("TextFieldMainClass");
    }

    private JTextField getTextFieldAddArgument() {
        return getTextField("TextFieldAddArgument");
    }

    private JButton getButtonAddArgument() {
        return (JButton) getButton("ButtonAddArgument");
    }

    private JButton getButtonRemoveArgument() {
        return (JButton) getButton("ButtonRemoveArgument");
    }

    private JButton getButtonStartApplication() {
        return (JButton) getButton("ButtonStartApplication");
    }

    private JLabel getLabelMainClass() {
        return getLabel("LabelMainClass");
    }

    private JLabel getLabelClassPath() {
        return getLabel("LabelClassPath");
    }

    private JLabel getLabelTitle() {
        return getLabel("LabelTitle");
    }
    
    private JScrollPane getScrollPaneClasspath() {
        return (JScrollPane) getComponentByName("ScrollPaneClasspath");
    }

    private JScrollPane getScrollPaneArguments() {
        return (JScrollPane) getComponentByName("ScrollPaneArguments");
    }

    private JList getListArguments() {
        return getList("ListArguments");
    }

    private void addJarFileClassPath() {
        final JFileChooser chooser = new JFileChooser();
        chooser.setCurrentDirectory(lastDir);
        chooser.setFileFilter(new FileFilter() {
            @Override
            public boolean accept(final File f) {
                return f.isDirectory() || f.getName().endsWith(".jar");
            }

            @Override
            public String getDescription() {
                return "Jar files (*.jar)";
            }
        });
        final int choice = chooser.showOpenDialog(this);
        if (choice != JFileChooser.CANCEL_OPTION) {
            classpathEntries.add(new ClasspathJarFileEntry(chooser
                    .getSelectedFile().getAbsolutePath()));
            lastDir = chooser.getSelectedFile().getParentFile();
        }
    }

    private void addBinDirClassPath() {
        final JDirectoryChooser chooser = new JDirectoryChooser();
        chooser.setCurrentDirectory(new File("."));
        final int choice = chooser.showOpenDialog(this);
        if (choice != JFileChooser.CANCEL_OPTION) {
            classpathEntries.add(new ClasspathBinDirEntry(chooser
                    .getSelectedFile().getAbsolutePath()));
            lastDir = chooser.getSelectedFile();
        }
    }

    private void addJarsDirClassPath() {
        final JDirectoryChooser chooser = new JDirectoryChooser();
        chooser.setCurrentDirectory(new File("."));
        final int choice = chooser.showOpenDialog(this);
        if (choice != JFileChooser.CANCEL_OPTION) {
            classpathEntries.add(new ClasspathJarsDirEntry(chooser
                    .getSelectedFile().getAbsolutePath()));
            lastDir = chooser.getSelectedFile();
        }
    }

    private void removeClassPath() {
        int row = getTableClassPath().getSelectedRow();
        classpathEntries.remove(row);
        final int rowCount = getTableClassPath().getRowCount();
        if (rowCount == 0) {
            repaint(getScrollPaneClasspath().getBounds());
        } else {
            if (row == rowCount) {
                row = row - 1;
            }
            getTableClassPath().getSelectionModel().setSelectionInterval(row,
                    row);
        }
        getTableClassPath().requestFocus();
    }

    private StartType getStartType() {
        final String name = (String) getComboboxStartType().getSelectedItem();
        return StartType.forName(name);
    }

    private void addArgument() {
        final String text = getTextFieldAddArgument().getText().trim();
        if (text.length() > 0) {
            arguments.add(text);
            getTextFieldAddArgument().setText("");
        } else {
            showError("Please enter something in the text field before pressing the add button!");
        }
    }

    private void removeArgument() {
        int idx = getListArguments().getSelectedIndex();
        arguments.remove(idx);
        final int size = arguments.size();
        if (size == 0) {
            repaint(getScrollPaneArguments().getBounds());
        } else {
            if (idx == size) {
                idx = idx - 1;
            }
            getListArguments().getSelectionModel().setSelectionInterval(idx,
                    idx);
        }
        getListArguments().requestFocus();
    }

    private void startApplication() {
        try {
            if (classpathEntries.size() == 0) {
                throw new StartApplicationException(
                        "Please add a classpath for your application in table '"
                                + getLabelClassPath().getText() + "'!");
            }
            if (getTextFieldMainClass().getText().length() == 0) {
                throw new StartApplicationException(
                        "Please enter something in field '"
                                + getLabelMainClass().getText() + "'!");
            }

            appStarter.start(getStartType(), classpathEntries,
                    getTextFieldMainClass().getText(), arguments);

        } catch (final StartApplicationException e) {
            showError(e.getMessage());
        }
    }

    private void showError(final String message) {
        JOptionPane.showMessageDialog(this, message, "Error",
                JOptionPane.ERROR_MESSAGE);
    }

}
