/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JSpinner;
import javax.swing.JTextField;

/**
 * Scales a <code>JSpinner</code>. This does not work if the spinner's editor
 * was set to an editor other than <code>JSpinner.DefaultEditor</code>.
 */
public final class JSpinnerScaler implements ComponentScaler {

    /**
     * {@inheritDoc}
     */
    public final Class<? extends Component> getType() {
        return JSpinner.class;
    }

    /**
     * {@inheritDoc}
     */
    public final ScaledComponent createScaledComponent(final Component comp,
            final ScalableLayoutRegistry registry) {
        return new ScaledComponent(comp, registry);
    }

    /**
     * {@inheritDoc}
     */
    public final void scale(final ScaledComponent sc, final double factor,
            final ScalableLayoutRegistry registry) {
        if (!(sc.getComp() instanceof JSpinner)) {
            throw new IllegalArgumentException(
                    "The argument 'sc' does not contain a 'JSpinner' component!");
        }

        // Scale the spinner itself
        final JSpinner spinner = (JSpinner) sc.getComp();
        spinner.setFont(ScalableLayoutUtils.scale(sc.getFont(), factor));
        if (sc.getBounds() == null) {
            // First call
            sc.setBounds(spinner.getBounds());
        }
        spinner.setBounds(ScalableLayoutUtils.scale(sc.getBounds(), factor));
        spinner.setPreferredSize(ScalableLayoutUtils.scale(sc.getPreferredSize(), factor));
        spinner.setMinimumSize(ScalableLayoutUtils.scale(sc.getMinimumSize(), factor));
        spinner.setMaximumSize(ScalableLayoutUtils.scale(sc.getMaximumSize(), factor));

        // The spinner contains a text field with a font
        // that needs to be scaled too
        final JComponent comp = spinner.getEditor();
        if (comp instanceof JSpinner.DefaultEditor) {
            final JSpinner.DefaultEditor defaultEditor = (JSpinner.DefaultEditor) comp;
            final JTextField tf = defaultEditor.getTextField();
            tf.setFont(ScalableLayoutUtils.scale(sc.getFont(), factor));
        }

    }

}
