/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.border.Border;

import org.fuin.utils4j.Utils4J;

/**
 * A layout manager that resizes the content along to a given factor.
 */
public abstract class AbstractScalableLayout implements LayoutManager {

    private double factor = 1;

    private final Font orgFont;

    private final Insets orgInsets;

    private final ScaledBorder orgBorder;

    private final LayoutManager orgLayout;

    private final Container container;

    private final List<ScaledComponent> components;

    private final ScalableLayoutRegistry registry;

    /**
     * Constructor with container.
     * 
     * @param registry
     *            Registry to use - Cannot be <code>null</code>.
     * @param container
     *            Container this layout should be applied to - Cannot be <code>null</code>.
     */
    protected AbstractScalableLayout(final ScalableLayoutRegistry registry,
            final Container container) {
        super();
        components = new ArrayList<ScaledComponent>();

        Utils4J.checkNotNull("registry", registry);        
        this.registry = registry;

        Utils4J.checkNotNull("container", container);        
        this.container = container;

        // Store container informations
        orgLayout = container.getLayout();
        orgFont = container.getFont();
        final Insets in = container.getInsets();
        orgInsets = new Insets(in.top, in.left, in.bottom, in.right);
        if (container instanceof JComponent) {
            final Border border = ((JComponent) container).getBorder();
            if (border == null) {
                orgBorder = null;
            } else {
                final BorderScaler scaler = registry.getBorderScaler(border.getClass());
                orgBorder = scaler.createScaledBorder(border, registry);
            }
        } else {
            orgBorder = null;
        }

        // Add all components to the internal list
        final Component[] comps = container.getComponents();
        for (int i = 0; i < comps.length; i++) {
            addScaledComponent(comps[i]);
        }
    }

    /**
     * {@inheritDoc}
     */
    public final void addLayoutComponent(final String name, final Component comp) {
        addScaledComponent(comp);
        if (orgLayout != null) {
            orgLayout.addLayoutComponent(name, comp);
        }
    }

    /**
     * Adds a component to the internal list of scaled components.
     * 
     * @param comp Component to add.
     */
    protected final void addScaledComponent(final Component comp) {
        final ComponentScaler scaler = registry.getComponentScaler(comp.getClass());
        final ScaledComponent sc = scaler.createScaledComponent(comp, registry);
        components.add(sc);
    }

    /**
     * {@inheritDoc}
     */
    public final void removeLayoutComponent(final Component comp) {
        final ScaledComponent sc = findScaledComponent(comp);
        if (sc != null) {
            components.remove(sc);
        }
        if (orgLayout != null) {
            orgLayout.removeLayoutComponent(comp);
        }
    }

    /**
     * Returns the corresponding scaled component.
     * 
     * @param comp
     *            Component to lookup a scaled component for.
     * 
     * @return Scaled component or <code>null</code> if not found.
     */
    protected final ScaledComponent findScaledComponent(final Component comp) {
        for (int i = 0; i < components.size(); i++) {
            final ScaledComponent sc = components.get(i);
            if (sc.getComp() == comp) {
                return sc;
            }
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public final Dimension preferredLayoutSize(final Container container) {
        if (orgLayout == null) {
            return null;
        }
        return orgLayout.preferredLayoutSize(container);
    }

    /**
     * {@inheritDoc}
     */
    public final Dimension minimumLayoutSize(final Container container) {
        if (orgLayout == null) {
            return null;
        }
        return orgLayout.minimumLayoutSize(container);
    }

    private void scaleComponents() {

        // Scale container
        if (container instanceof JComponent) {
            final JComponent jc = (JComponent) container;
            if (jc.getBorder() == null) {
                // Try to change insets because there is no border
                ScalableLayoutUtils.scale(container.getInsets(), orgInsets, factor);
            } else {
                final BorderScaler bs = registry.getBorderScaler(jc.getBorder().getClass());
                jc.setBorder(bs.scale(orgBorder, factor, registry));
            }
        } else {
            // Change insets only for AWT containers
            ScalableLayoutUtils.scale(container.getInsets(), orgInsets, factor);
        }
        container.setFont(ScalableLayoutUtils.scale(orgFont, factor));

        // Scale components
        final Component[] comps = container.getComponents();
        for (int i = 0; i < comps.length; i++) {
            final Component comp = comps[i];
            final ScaledComponent scaledComponent = findScaledComponent(comp);
            if (scaledComponent == null) {
                throw new IllegalStateException("Component not found: " + comp);
            }
            final ComponentScaler scaler = registry.getComponentScaler(scaledComponent
                    .getComp().getClass());
            scaler.scale(scaledComponent, factor, registry);
        }

        afterComponentsScaled(factor);

    }

    /**
     * Returns the current scaling factor.
     * 
     * @return Factor.
     */
    public final double getFactor() {
        return factor;
    }

    /**
     * Sets the scaling factor to a new value.
     * 
     * @param factor
     *            Factor to set.
     */
    public final void setFactor(final double factor) {
        this.factor = factor;
        scaleComponents();
        layoutContainer(container);
    }

    /**
     * Sets the scaling factor to a new value. If any of the components also has
     * a scalable layout it's <code>setFactorRecursive(double)</code> method
     * will be called too.
     * 
     * @param factor
     *            Factor to set.
     */
    public final void setFactorRecursive(final double factor) {

        // Scale components
        final Component[] comps = container.getComponents();
        for (int i = 0; i < comps.length; i++) {
            final Component comp = comps[i];
            setFactorRecursive(comp, factor);
        }

        // Scale this
        this.factor = factor;
        scaleComponents();
        layoutContainer(container);

    }

    /**
     * This method will be called after all components are scaled.
     * 
     * @param factor
     *            Factor used.
     */
    public abstract void afterComponentsScaled(final double factor);

    private static void setFactorRecursive(final Component comp, final double factor) {
        if (comp instanceof Container) {
            final Container cont = (Container) comp;
            if (cont instanceof JTabbedPane) {
                final JTabbedPane tabbedPane = (JTabbedPane) cont;
                for (int j = 0; j < tabbedPane.getComponentCount(); j++) {
                    final Component tabComponent = tabbedPane.getComponent(j);
                    setFactorRecursive(tabComponent, factor);
                }
            } else {
                if (cont.getLayout() instanceof AbstractScalableLayout) {
                    final AbstractScalableLayout layout = (AbstractScalableLayout) cont
                            .getLayout();
                    layout.setFactorRecursive(factor);
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public final void layoutContainer(final Container container) {
        if (orgLayout != null) {
            // Some layouts do check if container layout is identical with them
            container.setLayout(orgLayout);
            try {
                orgLayout.layoutContainer(container);
            } finally {
                container.setLayout(this);
            }
        }
    }

    /**
     * Returns the container this layout depends to.
     * 
     * @return Container.
     */
    protected final Container getContainer() {
        return container;
    }

}
