/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.glazedlists;

import java.util.List;
import java.util.Locale;

import org.fuin.objects4j.TableColumnInfo;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgMethod;

/**
 * Utility-Class to create a Table-Format class "on-the-fly" from
 * <code>@Label</code> and <code>@TableColumn</code> annotations. The
 * <code>TableFormatExt</code> implementation is created with "Javassist" - No
 * reflection is used after the class was constructed.
 * 
 * @see {@link org.fuin.objects4j.Label}
 * @see {@link org.fuin.objects4j.TableColumn}
 * @see {@link org.fuin.utils4swing.glazedlists.TableFormatExt}
 * 
 * @param <E>
 *            A business class annotated with <code>@TableColumn</code>.
 */
public final class TableFormatCreator<E> {

    private final ByteCodeGenerator generator;

    private final SgClassPool pool;

    /**
     * Default constructor.
     */
    public TableFormatCreator() {
        this(null, null);
    }

    /**
     * Constructor with class pool and generator.
     * 
     * @param pool
     *            Pool to use. If NULL a new instance will be created and used.
     * @param generator
     *            Generator to use. If NULL a new instance will be created and
     *            used.
     */
    public TableFormatCreator(final SgClassPool pool, final ByteCodeGenerator generator) {
        super();
        if (pool == null) {
            this.pool = new SgClassPool();
        } else {
            this.pool = pool;
        }
        if (generator == null) {
            this.generator = new ByteCodeGenerator();
        } else {
            this.generator = generator;
        }
    }

    /**
     * Creates a table format for the given class. The default
     * <code>Locale</code> is used to determine the <code>ResourceBundle</code>.
     * 
     * @param clasz
     *            Class annotated with <code>@Label</code> and
     *            <code>@TableColumn</code> annotations.
     * 
     * @return Table format.
     */
    public TableFormatExt<E> create(final Class<E> clasz) {
        return create(clasz, Locale.getDefault());
    }

    /**
     * Creates a table format for the given class.
     * 
     * @param clasz
     *            Class annotated with <code>@Label</code> and
     *            <code>@TableColumn</code> annotations.
     * @param locale
     *            Locale to use for determining the <code>ResourceBundle</code>.
     * 
     * @return Table format.
     */
    @SuppressWarnings("unchecked")
    public TableFormatExt<E> create(final Class<E> clasz, final Locale locale) {

        final String pkg = clasz.getSimpleName() + "TableFormat_" + locale;
        final String implName = clasz.getSimpleName() + "TableFormat_" + locale;
        Class<TableFormatExt<E>> tableFormatImpl = getClassSilent(pkg, implName);
        if (tableFormatImpl == null) {
            // Reflect annotations of target class
            final List<TableColumnInfo> columns = TableColumnInfo.create(clasz, locale);
            if (columns.size() == 0) {
                throw new IllegalArgumentException("No '@TableColumn' annotations found in class '"
                        + clasz.getName() + "'!");
            }

            final SgClass tableFormatImplClass = createTableFormatImplClass(pkg, implName);

            // int getColumnCount()
            addGetColumnCountMethod(tableFormatImplClass, columns);

            // String getColumnName(int)
            addGetColumnNameMethod(tableFormatImplClass, columns, clasz);

            // Object getColumnValue(Object, int)
            addGetColumnValueMethod(tableFormatImplClass, columns, clasz);

            // int getColumnWidth(int)
            addGetColumnWidthMethod(tableFormatImplClass, columns);

            tableFormatImpl = generator.createClass(tableFormatImplClass);
        }
        return (TableFormatExt<E>) generator.createInstance(tableFormatImpl);

    }

    @SuppressWarnings("unchecked")
    private Class<TableFormatExt<E>> getClassSilent(final String pkg, final String implName) {
        Class<TableFormatExt<E>> tableFormatImpl;
        try {
            tableFormatImpl = (Class<TableFormatExt<E>>) Class.forName(pkg + "." + implName);
            // The class already exists!
        } catch (ClassNotFoundException e) {
            tableFormatImpl = null;
        }
        return tableFormatImpl;
    }

    /**
     * Create a class implementing the <code>TableFormatExt</code> interface.
     * 
     * @param packageName
     *            Name of the destination package.
     * @param implName
     *            Name of the class to create.
     * 
     * @return Class object.
     * 
     */
    private SgClass createTableFormatImplClass(final String packageName, final String implName) {

        final String tableFormatInterfaceName = TableFormatExt.class.getName();
        final SgClass tableFormatImplClass = new SgClass(packageName, implName);
        final SgClass tableFormatInterface = SgClass.create(pool, tableFormatInterfaceName);
        tableFormatImplClass.addInterface(tableFormatInterface);

        return tableFormatImplClass;

    }

    /**
     * Add a <code>public int getColumnWidth(int)</code> method to the
     * <code>tableFormatImplClass</code> class.
     * 
     * @param tableFormatImplClass
     *            Class to add a method.
     * @param columns
     *            Columns.
     */
    private void addGetColumnWidthMethod(final SgClass tableFormatImplClass,
            final List<TableColumnInfo> columns) {

        // Create method with arguments
        final SgMethod getColumnWidthMethod = new SgMethod(tableFormatImplClass, "public",
                SgClass.INT, "getColumnWidth");
        // Adds the argument implicitly to the method
        new SgArgument(getColumnWidthMethod, SgClass.INT, "column");

        // Create and set source
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < columns.size(); i++) {
            if (i == 0) {
                sb.append("if (column == " + i + ") {");
            } else {
                sb.append("} else if (column == " + i + ") {");
            }
            sb.append("return " + columns.get(i).getWidth().toPixel() + ";");
        }
        sb.append("}");
        sb.append("throw new IllegalStateException(\"Unknown colum # \" + column);");
        getColumnWidthMethod.addBodyLine(sb.toString());

        // Add method to target class
        tableFormatImplClass.addMethod(getColumnWidthMethod);

    }

    /**
     * Add a <code>public Object getColumnValue(Object, int)</code> method to
     * the <code>tableFormatImplClass</code> class.
     * 
     * @param tableFormatImplClass
     *            Class to add a method.
     * @param columns
     *            Columns.
     * @param clasz
     *            Target class.
     */
    private void addGetColumnValueMethod(final SgClass tableFormatImplClass,
            final List<TableColumnInfo> columns, final Class<E> clasz) {

        // Create method with arguments
        final SgMethod getColumnValueMethod = new SgMethod(tableFormatImplClass, "public",
                SgClass.OBJECT, "getColumnValue");
        // Adds the arguments implicitly to the method
        new SgArgument(getColumnValueMethod, SgClass.OBJECT, "obj");
        new SgArgument(getColumnValueMethod, SgClass.INT, "column");

        // Create and set source
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < columns.size(); i++) {
            if (i == 0) {
                sb.append("if (column == " + i + ") {");
            } else {
                sb.append("} else if (column == " + i + ") {");
            }
            sb
                    .append("return ((" + clasz.getName() + ")obj)." + columns.get(i).getGetter()
                            + "();");
        }
        sb.append("}");
        sb.append("throw new IllegalStateException(\"Unknown colum # \" + column);");
        getColumnValueMethod.addBodyLine(sb.toString());

        // Add method to target class
        tableFormatImplClass.addMethod(getColumnValueMethod);

    }

    /**
     * Add a <code>public String getColumnName(int column)</code> method to the
     * <code>tableFormatImplClass</code> class.
     * 
     * @param tableFormatImplClass
     *            Class to add a method.
     * @param columns
     *            Columns.
     * @param clasz
     *            Target class.
     * 
     */
    private void addGetColumnNameMethod(final SgClass tableFormatImplClass,
            final List<TableColumnInfo> columns, final Class<E> clasz) {

        // Create method with arguments
        final SgMethod getColumnNameMethod = new SgMethod(tableFormatImplClass, "public", SgClass
                .create(pool, "java.lang.String"), "getColumnName");
        // Adds the argument implicitly to the method
        new SgArgument(getColumnNameMethod, SgClass.INT, "column");

        // Create and set source
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < columns.size(); i++) {
            if (i == 0) {
                sb.append("if (column == " + i + ") {");
            } else {
                sb.append("} else if (column == " + i + ") {");
            }
            sb.append("return \"" + columns.get(i).getText() + "\";");
        }
        sb.append("}");
        sb.append("throw new IllegalStateException(\"Unknown colum # \" + column);");
        getColumnNameMethod.addBodyLine(sb.toString());

        // Add method to target class
        tableFormatImplClass.addMethod(getColumnNameMethod);

    }

    /**
     * Add a <code>public int getColumnCount()</code> method to the
     * <code>tableFormatImplClass</code> class.
     * 
     * @param tableFormatImplClass
     *            Class to add a method.
     * @param columns
     *            Columns.
     * @param clasz
     *            Target class.
     * 
     */
    private void addGetColumnCountMethod(final SgClass tableFormatImplClass,
            final List<TableColumnInfo> columns) {

        // Create method with arguments
        final SgMethod getColumnCountMethod = new SgMethod(tableFormatImplClass, "public",
                SgClass.INT, "getColumnCount");

        // Create and set source
        getColumnCountMethod.addBodyLine("return " + columns.size() + ";");

        // Add method to target class
        tableFormatImplClass.addMethod(getColumnCountMethod);

    }

}
