/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;

import javax.swing.JComponent;

import org.fuin.utils4j.Utils4J;

/**
 * Helper class for scaled AWT components that don't have client properties.
 */
public final class ScaledComponent {

    private final Component comp;

    private final Font font;

    private final Dimension preferredSize;

    private final Dimension minimumSize;

    private final Dimension maximumSize;

    private final ScaledBorder border;

    private Rectangle bounds;

    /**
     * Constructor with component.
     * 
     * @param comp
     *            Component (not yet scaled) - Cannot be <code>null</code>.
     * @param registry
     *            Current registry - Cannot be <code>null</code>.
     * 
     */
    public ScaledComponent(final Component comp, final ScalableLayoutRegistry registry) {
        super();
        Utils4J.checkNotNull("comp", comp);        
        this.comp = comp;

        Utils4J.checkNotNull("registry", registry);        

        font = comp.getFont();
        bounds = null;
        if (comp.getPreferredSize() == null) {
            preferredSize = null;
        } else {
            preferredSize = new Dimension(comp.getPreferredSize());
        }
        if (comp.getMinimumSize() == null) {
            minimumSize = null;
        } else {
            minimumSize = new Dimension(comp.getMinimumSize());
        }
        if (comp.getMaximumSize() == null) {
            maximumSize = null;
        } else {
            maximumSize = new Dimension(comp.getMaximumSize());
        }

        if (comp instanceof JComponent) {
            final JComponent jc = (JComponent) comp;
            if (jc.getBorder() == null) {
                border = null;
            } else {
                final BorderScaler scaler = registry
                        .getBorderScaler(jc.getBorder().getClass());
                border = scaler.createScaledBorder(jc.getBorder(), registry);
            }
        } else {
            border = null;
        }
    }

    /**
     * Original component.
     * 
     * @return Component.
     */
    public final Component getComp() {
        return comp;
    }

    /**
     * Returns the original font.
     * 
     * @return Original font.
     */
    public final Font getFont() {
        return font;
    }

    /**
     * Returns the original bounds.
     * 
     * @return Original bounds.
     */
    public final Rectangle getBounds() {
        return bounds;
    }

    /**
     * Sets the original bounds.
     * 
     * @param rectangle
     *            Original bounds.
     */
    public final void setBounds(final Rectangle rectangle) {
        this.bounds = new Rectangle(rectangle);
    }

    /**
     * Returns the original preferred size.
     * 
     * @return Original preferred size.
     */
    public final Dimension getPreferredSize() {
        return preferredSize;
    }

    /**
     * Returns the original minimum size.
     * 
     * @return Original minimum size.
     */
    public final Dimension getMinimumSize() {
        return minimumSize;
    }

    /**
     * Returns the original maximum size.
     * 
     * @return Original maximum size.
     */
    public final Dimension getMaximumSize() {
        return maximumSize;
    }

    /**
     * Returns the helper for the border.
     * 
     * @return Border helper class or <code>null</code> if no border was set.
     */
    public final ScaledBorder getBorder() {
        return border;
    }

}
