/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.LayoutManager2;

/**
 * A layout manager that resizes the content along to a given factor.
 */
public abstract class ScalableLayout2 extends AbstractScalableLayout implements LayoutManager2 {

    private final LayoutManager2 originalLayout;

    /**
     * Constructor with registry and container.
     * 
     * @param registry
     *            Registry to use.
     * @param container
     *            Container this layout should be applied to.
     */
    public ScalableLayout2(final ScalableLayoutRegistry registry, final Container container) {
        super(registry, container);

        if (!(container.getLayout() instanceof LayoutManager2)) {
            throw new IllegalArgumentException(
                    "The layout manager of 'container' is not type 'LayoutManager2'! ["
                            + container.getLayout() + "]");
        }
        originalLayout = (LayoutManager2) container.getLayout();

    }

    /**
     * {@inheritDoc}
     */
    public final void addLayoutComponent(final Component comp, final Object constraints) {
        addScaledComponent(comp);
        originalLayout.addLayoutComponent(comp, constraints);
    }

    /**
     * {@inheritDoc}
     */
    public final Dimension maximumLayoutSize(final Container target) {
        return originalLayout.maximumLayoutSize(target);
    }

    /**
     * {@inheritDoc}
     */
    public final float getLayoutAlignmentX(final Container target) {
        return originalLayout.getLayoutAlignmentX(target);
    }

    /**
     * {@inheritDoc}
     */
    public final float getLayoutAlignmentY(final Container target) {
        return originalLayout.getLayoutAlignmentY(target);
    }

    /**
     * {@inheritDoc}
     */
    public final void invalidateLayout(final Container target) {
        if (originalLayout != null) {
            // Some layouts do check if container layout is identical with them
            getContainer().setLayout(originalLayout);
            try {
                originalLayout.invalidateLayout(getContainer());
            } finally {
                getContainer().setLayout(this);
            }
        }
    }

    /**
     * Returns the original layout manager.
     * 
     * @return Layout Manager 2.
     */
    public final LayoutManager2 getOriginalLayout() {
        return originalLayout;
    }

}
