/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.layout.scalable;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.LayoutManager;
import java.awt.LayoutManager2;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.border.Border;

/**
 * default registry for scalers.
 */
public final class DefaultScalableLayoutRegistry implements ScalableLayoutRegistry {

    private static final String NEW_INSTANCE_ERROR = "Cannot create a new scalabla layout manager instance!";

    private final List<ComponentScaler> componentScalerList;

    private final DefaultComponentScaler defaultComponentScaler;

    private final List<BorderScaler> borderScalerList;

    private final DefaultBorderScaler defaultBorderScaler;

    private final Map<Class<? extends LayoutManager2>, Class<? extends ScalableLayout2>> scalableLayouts2;

    private final Map<Class<? extends LayoutManager>, Class<? extends ScalableLayout>> scalableLayouts;

    /**
     * No argument constructor.
     */
    public DefaultScalableLayoutRegistry() {
        super();
        componentScalerList = new ArrayList<ComponentScaler>();
        componentScalerList.add(new JSpinnerScaler());
        componentScalerList.add(new JComponentScaler());
        defaultComponentScaler = new DefaultComponentScaler();

        borderScalerList = new ArrayList<BorderScaler>();
        borderScalerList.add(new TitledBorderScaler());
        borderScalerList.add(new LineBorderScaler());
        defaultBorderScaler = new DefaultBorderScaler();

        scalableLayouts2 = new HashMap<Class<? extends LayoutManager2>, Class<? extends ScalableLayout2>>();
        scalableLayouts2.put(BorderLayout.class, ScalableBorderLayout.class);

        scalableLayouts = new HashMap<Class<? extends LayoutManager>, Class<? extends ScalableLayout>>();
    }

    /**
     * Add a new scaler to the registry.
     * 
     * @param scaler
     *            Scaler to add.
     */
    public final void addScaler(final ComponentScaler scaler) {
        componentScalerList.add(scaler);
    }

    /**
     * Remove a scaler from the registry.
     * 
     * @param scaler
     *            Scaler to remove.
     * 
     * @return If the list contained the element <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean removeScaler(final ComponentScaler scaler) {
        return componentScalerList.remove(scaler);
    }

    /**
     * {@inheritDoc}
     */
    public final ComponentScaler getComponentScaler(
            final Class<? extends Component> componentClass) {
        for (int i = 0; i < componentScalerList.size(); i++) {
            final ComponentScaler scaler = componentScalerList.get(i);
            if (scaler.getType().isAssignableFrom(componentClass)) {
                return scaler;
            }
        }
        return defaultComponentScaler;
    }

    /**
     * Add a new scaler to the registry.
     * 
     * @param scaler
     *            Scaler to add.
     */
    public final void addScaler(final BorderScaler scaler) {
        borderScalerList.add(scaler);
    }

    /**
     * Remove a scaler from the registry.
     * 
     * @param scaler
     *            Scaler to remove.
     * 
     * @return If the list contained the element <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean removeScaler(final BorderScaler scaler) {
        return borderScalerList.remove(scaler);
    }

    /**
     * {@inheritDoc}
     */
    public final BorderScaler getBorderScaler(final Class<? extends Border> borderClass) {
        for (int i = 0; i < borderScalerList.size(); i++) {
            final BorderScaler scaler = borderScalerList.get(i);
            if (scaler.getType().isAssignableFrom(borderClass)) {
                return scaler;
            }
        }
        return defaultBorderScaler;
    }

    /**
     * Check if the layout class has exact <code>ScalableLayoutRegistry</code>
     * and <code>Container</code> arguments.
     * 
     * @param layoutClass
     *            Class to check.
     */
    private void checkAbstractLayoutConstructor(
            final Class<? extends AbstractScalableLayout> layoutClass) {

        try {
            layoutClass.getConstructor(ScalableLayoutRegistry.class, Container.class);
            // OK, do nothing
        } catch (SecurityException ex) {
            throw new IllegalStateException("Class loader or security problem!", ex);
        } catch (NoSuchMethodException ex) {
            throw new IllegalArgumentException(
                    "The class must have a constructor with exact two arguments: "
                            + "'ScalableLayoutRegistry' and 'Container'(in this order)!", ex);
        }

    }

    /**
     * Adds a layout manager class mapping to the registry.
     * 
     * @param layoutManager2Class
     *            Layout manager 2 to replace with
     *            <code>scalableLayout2Class</code>.
     * @param scalableLayout2Class
     *            Scalable layout manager class to add. The class must have a
     *            constructor with exact two arguments:
     *            <code>ScalableLayoutRegistry</code> and <code>Container</code>
     *            (in this order).
     */
    public final void putScalableLayout2(
            final Class<? extends LayoutManager2> layoutManager2Class,
            final Class<? extends ScalableLayout2> scalableLayout2Class) {
        checkAbstractLayoutConstructor(scalableLayout2Class);
        scalableLayouts2.put(layoutManager2Class, scalableLayout2Class);
    }

    /**
     * Removes a layout manager class mapping from the registry.
     * 
     * @param layoutManager2Class
     *            Scalable layout manager class to remove.
     */
    public final void removeScalableLayout2(
            final Class<? extends LayoutManager2> layoutManager2Class) {
        scalableLayouts2.remove(layoutManager2Class);
    }

    /**
     * Adds a layout manager class mapping to the registry.
     * 
     * @param layoutManagerClass
     *            Layout manager to replace with
     *            <code>scalableLayoutClass</code>.
     * @param scalableLayoutClass
     *            Scalable layout manager class to add. The class must have a
     *            constructor with exact two arguments:
     *            <code>ScalableLayoutRegistry</code> and <code>Container</code>
     *            (in this order).
     */
    public final void putScalableLayout(
            final Class<? extends LayoutManager> layoutManagerClass,
            final Class<? extends ScalableLayout> scalableLayoutClass) {
        checkAbstractLayoutConstructor(scalableLayoutClass);
        scalableLayouts.put(layoutManagerClass, scalableLayoutClass);
    }

    /**
     * Removes a layout manager class mapping from the registry.
     * 
     * @param layoutManagerClass
     *            Layout manager class to remove.
     */
    public final void removeScalableLayout(
            final Class<? extends LayoutManager> layoutManagerClass) {
        scalableLayouts.remove(layoutManagerClass);
    }

    /**
     * Creates a new scalable layout instance.
     * 
     * @param clasz
     *            Layout class.
     * @param registry
     *            Registry.
     * @param container
     *            Container.
     * 
     * @return New instance.
     */
    @SuppressWarnings("unchecked")
    private AbstractScalableLayout createScalableLayoutInstance(
            final Class<? extends AbstractScalableLayout> clasz,
            final ScalableLayoutRegistry registry, final Container container) {

        try {
            final Constructor constructor = clasz.getConstructor(ScalableLayoutRegistry.class,
                    Container.class);
            return (AbstractScalableLayout) constructor.newInstance(registry, container);
        } catch (final SecurityException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        } catch (final NoSuchMethodException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        } catch (final IllegalArgumentException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        } catch (final InstantiationException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        } catch (final IllegalAccessException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        } catch (final InvocationTargetException ex) {
            throw new RuntimeException(NEW_INSTANCE_ERROR, ex);
        }

    }

    /**
     * {@inheritDoc}
     */
    public final AbstractScalableLayout getScalableLayout(
            final ScalableLayoutRegistry registry, final Container container) {

        final LayoutManager originalLayout = container.getLayout();

        final AbstractScalableLayout layout;
        if (originalLayout instanceof LayoutManager2) {
            Class<? extends ScalableLayout2> layoutClass = null;
            final Iterator<Class<? extends LayoutManager2>> it = scalableLayouts2.keySet()
                    .iterator();
            while (it.hasNext()) {
                final Class<? extends LayoutManager2> clasz = it.next();
                if (clasz.isAssignableFrom(originalLayout.getClass())) {
                    layoutClass = scalableLayouts2.get(clasz);
                    break;
                }
            }
            if (layoutClass == null) {
                layoutClass = DefaultScalableLayout2.class;
            }
            layout = createScalableLayoutInstance(layoutClass, registry, container);
        } else {
            if (originalLayout == null) {
                layout = new DefaultScalableLayout(registry, container);
            } else {
                Class<? extends ScalableLayout> layoutClass = null;
                final Iterator<Class<? extends LayoutManager>> it = scalableLayouts.keySet()
                        .iterator();
                while (it.hasNext()) {
                    final Class<? extends LayoutManager> clasz = it.next();
                    if (clasz.isAssignableFrom(originalLayout.getClass())) {
                        layoutClass = scalableLayouts.get(clasz);
                        break;
                    }
                }
                if (layoutClass == null) {
                    layoutClass = DefaultScalableLayout.class;
                }
                layout = createScalableLayoutInstance(layoutClass, registry, container);
            }
        }

        return layout;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isContainer(final Component container) {
        if (container instanceof JPanel) {
            return true;
        } else if (container instanceof JTabbedPane) {
            return true;
        }
        return false;
    }

}
