/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.appstarter;

import java.applet.Applet;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.appletviewer.SimpleAppletViewer;
import org.fuin.utils4swing.appletviewer.SimpleAppletViewerConfig;
import org.fuin.utils4swing.common.ScreenCenterPositioner;
import org.fuin.utils4swing.common.Utils4Swing;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;

/**
 * Starts any application by injecting single jars, directories with jars or
 * classpath directories into the classpath.
 */
public final class AppStarter {

    private final AppStarterConfig config;

    private final AppStarterListener listener;

    private JFrame configFrame;

    /**
     * Constructor with configuration.
     * 
     * @param config
     *            Configuration.
     */
    public AppStarter(final AppStarterConfig config) {
        this(config, null);
    }

    /**
     * Constructor with configuration and listener.
     * 
     * @param config
     *            Configuration - Cannot be <code>null</code>.
     * @param listener
     *            Listener to be notified when the application started.
     */
    public AppStarter(final AppStarterConfig config, final AppStarterListener listener) {
        super();
        Utils4J.checkNotNull("config", config);        
        this.config = config;
        this.listener = listener;
    }

    /**
     * Injects entries in the classpath and starts the application.
     * 
     * @param startType
     *            Start an application or an applet?
     * @param classpathEntries
     *            Entries to inject into the classpath.
     * @param mainClass
     *            Main class to start (applet class or class with main method).
     * @param arguments
     *            Arguments for the application or the applet. Arguments for an
     *            applet must have the form "&lt;key&gt;=&lt;value&gt;".
     * 
     * @throws StartApplicationException
     *             Application failed to start.
     */
    public final void start(final StartType startType,
            final List<ClasspathEntry> classpathEntries, final String mainClass,
            final List<String> arguments) throws StartApplicationException {

        start(startType, classpathEntries, mainClass, arguments, listener);
        configFrame.setVisible(false);

    }

    /**
     * Injects entries in the classpath and starts the application.
     * 
     * @param startType
     *            Start an application or an applet?
     * @param classpathEntries
     *            Entries to inject into the classpath.
     * @param mainClass
     *            Main class to start (applet class or class with main method).
     * @param arguments
     *            Arguments for the application or the applet. Arguments for an
     *            applet must have the form "&lt;key&gt;=&lt;value&gt;".
     * @param listener
     *            Listener to be notified when the target application started.
     * 
     * @throws StartApplicationException
     *             Application failed to start.
     */
    @SuppressWarnings("unchecked")
    public static final void start(final StartType startType,
            final List<ClasspathEntry> classpathEntries, final String mainClass,
            final List<String> arguments, final AppStarterListener listener)
            throws StartApplicationException {

        for (int i = 0; i < classpathEntries.size(); i++) {
            classpathEntries.get(i).addToClasspath();
        }

        try {
            final Class clasz = Class.forName(mainClass);
            if (startType.equals(StartType.APPLET)) {
                startApplet(mainClass, arguments, clasz);
            } else {
                startApplication(mainClass, arguments, clasz);
            }

            if (listener != null) {
                listener.appStarted();
            }

        } catch (final ClassNotFoundException e) {
            throw new StartApplicationException("Main class '" + mainClass
                    + "' was not found in classpath!", e);
        }

    }

    @SuppressWarnings("unchecked")
    private static void startApplet(final String mainClass, final List<String> arguments,
            final Class clasz) throws StartApplicationException {

        if (!Applet.class.isAssignableFrom(clasz)) {
            throw new StartApplicationException("Main class '" + mainClass
                    + "' is not an Applet!");
        }

        final SimpleAppletViewerConfig appletViewerConfig = new SimpleAppletViewerConfig();
        appletViewerConfig.setApplet(mainClass);
        appletViewerConfig.setArguments(arguments);

        (new SimpleAppletViewer(appletViewerConfig)).execute();

    }

    @SuppressWarnings("unchecked")
    private static void startApplication(final String mainClass, final List<String> arguments,
            final Class clasz) throws StartApplicationException {
        try {

            final Method method = clasz.getMethod("main", String[].class);
            final String[] args = arguments.toArray(new String[arguments.size()]);
            method.invoke(clasz, (Object) args);
        } catch (final SecurityException e) {
            e.printStackTrace();
            throw new StartApplicationException(
                    "SecurityException getting 'main(String[])' method!", e);
        } catch (final NoSuchMethodException e) {
            e.printStackTrace();
            throw new StartApplicationException("Main class '" + mainClass
                    + "' has no 'main(String[])' method!", e);
        } catch (final IllegalArgumentException e) {
            e.printStackTrace();
            throw new StartApplicationException(
                    "IllegalArgumentException calling 'main(String[])' method!", e);
        } catch (final IllegalAccessException e) {
            e.printStackTrace();
            throw new StartApplicationException(
                    "IllegalAccessException calling 'main(String[])' method!", e);
        } catch (final InvocationTargetException e) {
            e.printStackTrace();
            throw new StartApplicationException(
                    "InvocationTargetException calling 'main(String[])' method!", e);
        }
    }

    /**
     * Starts the application.
     */
    public final void execute() {
        configFrame = Utils4Swing.createShowAndPosition(config.getFrameTitle(),
                new AppStarterConfigPanel(config, this), true, false,
                new ScreenCenterPositioner());
    }

    /**
     * Starts the application from the command line.
     * 
     * @param args
     *            Command line arguments.
     * 
     * @throws Exception
     *             Start failed.
     */
    public static void main(final String[] args) throws Exception {
        final AppStarterConfig config = new AppStarterConfig();
        final CmdLineParser parser = new CmdLineParser(config);
        try {
            parser.parseArgument(args);

            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    Utils4Swing.initSystemLookAndFeel();
                    (new AppStarter(config)).execute();
                }
            });

        } catch (final CmdLineException e) {
            System.err.println(e.getMessage());
            System.err.println("java SwingRecorder [options...]");
            parser.printUsage(System.err);
            System.err.println();
            return;
        }
    }

}
