/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.dialogs;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import org.fuin.utils4swing.common.ScreenCenterPositioner;
import org.fuin.utils4swing.common.Utils4Swing;

/**
 * Helper class for selecting a directory. Can used from a non-EDT thread.
 */
public class DirectorySelector {

    private JFrame frame = null;

    private DirectorySelectionPanel panel = null;

    private boolean firstTime = true;

    private final String title;

    private final DirectorySelectorListener listener;

    private final String directory;

    /**
     * Constructor with title and start directory.
     * 
     * @param title
     *            Title of the frame and panel.
     * @param directory
     *            Start directory.
     * @param listener
     *            Listener to inform about the result.
     */
    public DirectorySelector(final String title, final String directory,
            final DirectorySelectorListener listener) {
        super();

        if (title == null) {
            throw new IllegalArgumentException("The argument 'title' cannot be null!");
        }
        this.title = title;

        if (directory == null) {
            throw new IllegalArgumentException("The argument 'directory' cannot be null!");
        }
        this.directory = directory;

        if (listener == null) {
            throw new IllegalArgumentException("The argument 'listener' cannot be null!");
        }
        this.listener = listener;

    }

    /**
     * Show the monitor dialog.If called outside the EDT this method will switch
     * to the UI thread using
     * <code>SwingUtilities.invokeAndWait(Runnable)</code>.
     */
    public final void show() {
        if (firstTime) {
            firstTime = false;
            if (SwingUtilities.isEventDispatchThread()) {
                showIntern();
            } else {
                try {
                    SwingUtilities.invokeAndWait(new Runnable() {
                        public void run() {
                            showIntern();
                        }
                    });
                } catch (final Exception ex) {
                    ignore();
                }
            }
        } else {
            throw new IllegalStateException("This object cannot be reused!");
        }
    }

    private final void showIntern() {
        panel = new DirectorySelectionPanel(this);
        panel.setTitle(title);
        panel.setDirectory(directory);
        frame = Utils4Swing.createShowAndPosition(title, panel, false,
                new ScreenCenterPositioner());
        frame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        frame.addWindowListener(new WindowAdapter() {
            @Override
            public final void windowClosing(final WindowEvent e) {
                cancel();
            }
        });
    }

    /**
     * Close the frame and free all resources. After calling "close" the monitor
     * object cannot be used again!If called outside the EDT this method will
     * switch to the UI thread using
     * <code>SwingUtilities.invokeAndWait(Runnable)</code>.
     */
    protected final void close() {
        if (SwingUtilities.isEventDispatchThread()) {
            closeIntern();
        } else {
            try {
                SwingUtilities.invokeAndWait(new Runnable() {
                    public void run() {
                        closeIntern();
                    }
                });
            } catch (final Exception ex) {
                ignore();
            }
        }
    }

    private final void closeIntern() {
        if (frame != null) {
            frame.setVisible(false);
            frame.dispose();
            panel = null;
            frame = null;
        }
    }

    /**
     * The user canceled the dialog.
     */
    protected final void cancel() {
        close();
        listener.canceled();
    }

    /**
     * The dialog was successfully finished.
     */
    protected final void ok() {
        final String dir = panel.getDirectory();
        close();
        listener.finished(dir);
    }

    private static void ignore() {
        // Dummy method to satisfy Checkstyle's empty block check...
    }

    /**
     * Blocks the current thread and waits for the result of the selection
     * dialog.
     * 
     * @param title
     *            Title to display for the dialog.
     * @param directory
     *            Initial directory.
     * 
     * @return Selected directory.
     * 
     * @throws CanceledException
     *             The user canceled the dialog.
     */
    public static String selectDirectory(final String title, final String directory)
            throws CanceledException {

        /**
         * Gets informed about the result.
         */
        class MyListener implements DirectorySelectorListener {

            private volatile String directory = null;

            /**
             * Selected directory.
             * 
             * @return Directory.
             */
            public String getDirectory() {
                return directory;
            }

            /**
             * {@inheritDoc}
             */
            public void canceled() {
                this.directory = "";
            }

            /**
             * {@inheritDoc}
             */
            public void finished(final String directory) {
                this.directory = directory;
            }
        }

        final MyListener listener = new MyListener();
        new DirectorySelector(title, directory, listener).show();

        // Wait for the result
        while (listener.getDirectory() == null) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                ignore();
            }
        }
        
        if (listener.getDirectory().length() == 0) {
            throw new CanceledException();
        }

        return listener.getDirectory();

    }

    /**
     * Starts the selector for a test.
     * 
     * @param args
     *            Not used.
     * 
     * @throws IOException
     *             Error creating the canonical path for the current directory.
     */
    public static void main(final String[] args) throws IOException {
        Utils4Swing.initSystemLookAndFeel();
        final File file = new File(".");
        final DirectorySelector selector = new DirectorySelector(
                "Please select the destination directory:", file.getCanonicalPath(),
                new DirectorySelectorListener() {
                    public void canceled() {
                        System.out.println("CANCELED!");
                    }

                    public void finished(final String directory) {
                        System.out.println("SELECTED=" + directory);
                    }
                });
        selector.show();
    }

}
