/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.appletviewer;

import java.applet.Applet;
import java.awt.Dimension;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import org.fuin.utils4swing.common.ScreenCenterPositioner;
import org.fuin.utils4swing.common.Utils4Swing;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;

/**
 * A simple replacement for the Sun Applet Viewer.
 */
public final class SimpleAppletViewer {

	private final SimpleAppletViewerConfig config;

	private final Applet applet;

	/**
	 * Constructor with configuration.
	 * 
	 * @param config
	 *            Configuration.
	 */
	public SimpleAppletViewer(final SimpleAppletViewerConfig config) {
		super();
		if (config == null) {
			throw new IllegalArgumentException(
					"The argument 'config' cannot be null!");
		}
		this.config = config;

		applet = config.getApplet();
		if (applet == null) {
			throw new IllegalArgumentException(
					"The 'applet' field in the config cannot be null!");
		}

	}

	/**
	 * Starts the viewer.
	 */
	public final void execute() {

		final String appletName;
		if (config.getName() == null) {
			appletName = applet.getClass().getSimpleName();
		} else {
			appletName = config.getName();
		}

		applet.setPreferredSize(new Dimension(config.getWidth(), config
				.getHeight()));

		final JFrame viewerFrame = Utils4Swing.createShowAndPosition(config
				.getTitle()
				+ ": " + appletName, applet, true, true,
				new ScreenCenterPositioner());

		final SimpleAppletContext appletContext = new SimpleAppletContext(
				applet, appletName);
		final SimpleAppletStub appletStub = new SimpleAppletStub(appletContext,
				config.getParams(), config.getBaseUrl());

		viewerFrame.addComponentListener(new ComponentListener() {
			public void componentHidden(final ComponentEvent e) {
			}

			public void componentMoved(final ComponentEvent e) {
			}

			public void componentResized(final ComponentEvent e) {
				appletStub.appletResize(viewerFrame.getWidth(), viewerFrame
						.getHeight());
			}

			public void componentShown(final ComponentEvent e) {
			}
		});

		applet.setStub(appletStub);
		applet.init();
		applet.start();

	}

	/**
	 * Starts the viewer via the command line.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(final String[] args) {
		final SimpleAppletViewerConfig config = new SimpleAppletViewerConfig();
		final CmdLineParser parser = new CmdLineParser(config);
		try {
			parser.parseArgument(args);

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					Utils4Swing.initSystemLookAndFeel();
					(new SimpleAppletViewer(config)).execute();
				}
			});

		} catch (final CmdLineException e) {
			System.err.println(e.getMessage());
			System.err
					.println("java SimpleAppletViewer [options] [Applet arguments]");
			parser.printUsage(System.err);
			System.err.println();
			return;
		}
	}

}
