/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.common;

import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.Insets;
import java.awt.KeyboardFocusManager;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.RootPaneContainer;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

/**
 * Common utility methods for use in Swing applications and libraries.
 */
public final class Utils4Swing {

	/**
	 * Private default constructor.
	 */
	private Utils4Swing() {
		throw new UnsupportedOperationException(
				"This utility class is not intended to be instanciated!");
	}

	/**
	 * Create a new resizeable frame with a panel as it's content pane and
	 * position the frame.
	 * 
	 * @param title
	 *            Frame title.
	 * @param content
	 *            Content.
	 * @param exitOnClose
	 *            Exit the program on closing the frame?
	 * @param positioner
	 *            FramePositioner.
	 * 
	 * @return A visible frame at the preferred position.
	 */
	public static JFrame createShowAndPosition(final String title, final Container content,
			final boolean exitOnClose, final FramePositioner positioner) {
		return createShowAndPosition(title, content, exitOnClose, true, positioner);
	}

	/**
	 * 
	 * @param title
	 *            Frame title.
	 * @param content
	 *            Content.
	 * @param exitOnClose
	 *            Exit the program on closing the frame?
	 * @param resizable
	 *            If the frame should be resizeable TRUE else FALSE.
	 * @param positioner
	 *            FramePositioner.
	 * 
	 * @return A visible frame at the preferred position.
	 */
	public static JFrame createShowAndPosition(final String title, final Container content,
			final boolean exitOnClose, final boolean resizable,
			final FramePositioner positioner) {

		final JFrame frame = new JFrame(title);
		frame.setContentPane(content);
		if (exitOnClose) {
			frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		} else {
			frame.setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
		}

		frame.setSize(content.getPreferredSize());

		positioner.position(frame);

		final Insets insets = frame.getInsets();

		frame.setSize(frame.getWidth() + insets.left + insets.right, frame.getHeight()
				+ insets.top + insets.bottom);

		frame.setVisible(true);
		frame.setResizable(resizable);

		return frame;

	}

	/**
	 * Returns the package path of a class.
	 * 
	 * @param clasz
	 *            Class to determine the path for.
	 * 
	 * @return Package path for the class.
	 */
	@SuppressWarnings("unchecked")
	public static String getPackagePath(final Class clasz) {
		return clasz.getPackage().getName().replace('.', '/');
	}

	/**
	 * Get the path to a resource located in the same package as a given class.
	 * 
	 * @param clasz
	 *            Class with the same package where the resource is located.
	 * @param name
	 *            Filename of the resource.
	 * 
	 * @return Resource URL.
	 */
	@SuppressWarnings("unchecked")
	public static URL getResource(final Class clasz, final String name) {
		final String nameAndPath = getPackagePath(clasz) + "/" + name;
		return clasz.getResource(nameAndPath);
	}

	/**
	 * Load an icon located in the same package as a given class.
	 * 
	 * @param clasz
	 *            Class with the same package where the icon is located.
	 * @param name
	 *            Filename of the icon.
	 * 
	 * @return New icon instance.
	 */
	@SuppressWarnings("unchecked")
	public static ImageIcon loadIcon(final Class clasz, final String name) {
		final URL url = getResource(clasz, name);
		return new ImageIcon(url);
	}

	/**
	 * Initializes the look and feel and wraps exceptions into a runtime
	 * exception.
	 * 
	 * @param className
	 *            Full qualified name of the look and feel class.
	 */
	public static void initLookAndFeel(final String className) {
		try {
			UIManager.setLookAndFeel(className);
		} catch (final Exception e) {
			throw new RuntimeException("Error initializing the Look And Feel!", e);
		}
	}

	/**
	 * Initializes the system look and feel and wraps exceptions into a runtime
	 * exception.
	 */
	public static void initSystemLookAndFeel() {
		initLookAndFeel(UIManager.getSystemLookAndFeelClassName());
	}

	/**
	 * Create an instance with Class.forName(..) and wrap all exceptions into
	 * RuntimeExceptions.
	 * 
	 * @param className
	 *            Full qualified class name.
	 * 
	 * @return New instance of the class.
	 */
	@SuppressWarnings("unchecked")
	public static Object createInstance(final String className) {
		try {
			final Class clasz = Class.forName(className);
			return clasz.newInstance();
		} catch (final ClassNotFoundException e) {
			throw new RuntimeException("Unknown class!", e);
		} catch (final InstantiationException e) {
			throw new RuntimeException("Error instanciating class!", e);
		} catch (final IllegalAccessException e) {
			throw new RuntimeException("Error accessing class!", e);
		}
	}

	/**
	 * Adds an URL to the classpath.
	 * 
	 * @param url
	 *            URL to add.
	 */
	public static void addToClasspath(final String url) {
		try {
			addToClasspath(new URL(url));
		} catch (final MalformedURLException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Checks if the array or URLs contains the given URL.
	 * 
	 * @param urls
	 *            Array of URLs.
	 * @param url
	 *            URL to find.
	 * 
	 * @return If the URL is in the array TRUE else FALSE.
	 */
	public static boolean containsURL(final URL[] urls, final URL url) {
		for (final URL element : urls) {
			if (element.equals(url)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Adds an URL to the classpath.
	 * 
	 * @param url
	 *            URL to add.
	 */
	public static void addToClasspath(final URL url) {
		final ClassLoader classLoader = Utils4Swing.class.getClassLoader();
		if (!(classLoader instanceof URLClassLoader)) {
			throw new IllegalArgumentException("Cannot add '" + url
					+ "' to classloader because it's not an URL classloader");
		}
		final URLClassLoader urlClassLoader = (URLClassLoader) classLoader;
		if (!containsURL(urlClassLoader.getURLs(), url)) {
			try {

				final Method addURL = URLClassLoader.class.getDeclaredMethod("addURL",
						new Class[] { URL.class });
				addURL.setAccessible(true);
				addURL.invoke(urlClassLoader, new Object[] { url });
			} catch (final NoSuchMethodException e) {
				throw new RuntimeException(e);
			} catch (final IllegalArgumentException e) {
				throw new RuntimeException(e);
			} catch (final IllegalAccessException e) {
				throw new RuntimeException(e);
			} catch (final InvocationTargetException e) {
				throw new RuntimeException(e);
			}
		}
	}

	/**
	 * Find the root pane container in the current hierarchy.
	 * 
	 * @param source
	 *            Component to start with.
	 * 
	 * @return Root pane container or NULL if it cannot be found.
	 */
	public static RootPaneContainer findRootPaneContainer(final Component source) {
		Component comp = source;
		while ((comp != null) && !(comp instanceof RootPaneContainer)) {
			comp = comp.getParent();
		}
		if (comp instanceof RootPaneContainer) {
			return (RootPaneContainer) comp;
		}
		return null;
	}

	public static GlassPaneState showGlassPane(final Component source) {
		final RootPaneContainer rootPaneContainer = findRootPaneContainer(source);
		final Component glassPane = rootPaneContainer.getGlassPane();
		final MouseListener mouseListener = new MouseAdapter() {
		};
		final Cursor cursor = glassPane.getCursor();
		glassPane.addMouseListener(mouseListener);
		glassPane.setVisible(true);
		glassPane.requestFocus();
		glassPane.setCursor(new Cursor(Cursor.WAIT_CURSOR));
		return new GlassPaneState(glassPane, mouseListener, KeyboardFocusManager
				.getCurrentKeyboardFocusManager().getFocusOwner(), cursor);
	}

	public static void hideGlassPane(final GlassPaneState state) {
		final Component glassPane = state.getGlassPane();
		glassPane.removeMouseListener(state.getMouseListener());
		glassPane.setCursor(state.getCursor());
		glassPane.setVisible(false);
		if (state.getFocusOwner() != null) {
			state.getFocusOwner().requestFocus();
		}
	}

	public static final class GlassPaneState {

		private final Component glassPane;

		private final MouseListener mouseListener;

		private final Component focusOwner;
		
		private final Cursor cursor;

		public GlassPaneState(final Component glassPane, final MouseListener mouseListener,
				final Component focusOwner, final Cursor cursor) {
			super();
			this.glassPane = glassPane;
			this.mouseListener = mouseListener;
			this.focusOwner = focusOwner;
			this.cursor = cursor;
		}

		public Component getGlassPane() {
			return glassPane;
		}

		public MouseListener getMouseListener() {
			return mouseListener;
		}

		public Component getFocusOwner() {
			return focusOwner;
		}

		public Cursor getCursor() {
			return cursor;
		}

	}

}
