/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.appstarter;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.kohsuke.args4j.Option;

/**
 * Configuration for the "AppStarter".
 */
public class AppStarterConfig {

    /** Entries to add to the classpath. */
    private final List<ClasspathEntry> classPath;

    private final List<String> arguments;

    private StartType startType = StartType.APPLICATION;

    private String mainClass = "";

    /** Title used for the frame. */
    private String frameTitle = "AppStarter";

    /**
     * Constructor with no arguments.
     */
    public AppStarterConfig() {
        super();
        this.classPath = new ArrayList<ClasspathEntry>();
        this.arguments = new ArrayList<String>();
    }

    /**
     * Returns a list of arguments to use for the application.
     * 
     * @return Argument list.
     */
    public final List<String> getArguments() {
        return arguments;
    }

    /**
     * Set one or more arguments for the application.
     * 
     * @param args Arguments separated by comma.
     */
    @Option(name = "-arguments", usage = "Sets one or more arguments separated by a comma")
    public final void setArguments(final String args) {
        final StringTokenizer tok = new StringTokenizer(args, ",");
        while (tok.hasMoreTokens()) {
            arguments.add(tok.nextToken());
        }
    }
    
    /**
     * Adds a command line argument.
     * 
     * @param arg Argument to add.
     */
    public final void addArgument(final String arg) {
    	arguments.add(arg);
    }

    /**
     * Returns a list of entries to inject into the classpath.
     * 
     * @return List of classpath entries.
     */
    public final List<ClasspathEntry> getClassPath() {
        return classPath;
    }

    /**
     * Adds a single classpath entry to the list.
     * 
     * @param entry Entry to add.
     */
    public final void addClasspathEntry(final ClasspathEntry entry) {
    	classPath.add(entry);
    }
    
    /**
     * Returns a list of entries to inject into the classpath.
     * 
     * @param cp Classpath entries separated by a comma and with format
     *            'TYPE|VALUE'. Valid types are: CLASSES_DIR, JAR_FILE and
     *            JARS_DIR.
     */
    @Option(name = "-classpathEntries", usage = "Sets one or more classpath separated by a "
            + "komma and with format 'TYPE|VALUE'")
    public final void setClassPath(final String cp) {
        final StringTokenizer tok = new StringTokenizer(cp, ",");
        while (tok.hasMoreTokens()) {
            final String str = tok.nextToken();
            if (str.startsWith(ClasspathJarFileEntry.TYPE + "|")) {
                classPath.add(new ClasspathJarFileEntry(str
                        .substring(ClasspathJarFileEntry.TYPE.length() + 1)));
            } else if (str.startsWith(ClasspathJarsDirEntry.TYPE + "|")) {
                classPath.add(new ClasspathJarsDirEntry(str
                        .substring(ClasspathJarsDirEntry.TYPE.length() + 1)));
            } else if (str.startsWith(ClasspathBinDirEntry.TYPE + "|")) {
                classPath.add(new ClasspathBinDirEntry(str
                        .substring(ClasspathBinDirEntry.TYPE.length() + 1)));
            } else {
                throw new IllegalArgumentException("The classpath type for '"
                        + str + "' is unknown!");
            }
        }
    }

    /**
     * Returns the class to start.
     * 
     * @return Full qualified class name (Applet or class with main method).
     */
    public final String getMainClass() {
        return mainClass;
    }

    /**
     * Sets the class to start.
     * 
     * @param mainClass Full qualified class name (Applet or class with main
     *            method).
     */
    @Option(name = "-mainClass", usage = "Sets the main class to start")
    public final void setMainClass(final String mainClass) {
        this.mainClass = mainClass;
    }

    /**
     * Returns the type of application to start.
     * 
     * @return Applet or Application.
     */
    public final StartType getStartType() {
        return startType;
    }

    /**
     * Sets the type of application to start.
     * 
     * @param startType Applet or Application.
     */
    @Option(name = "-startType", usage = "Sets the start type ('Application' or 'Applet')")
    public final void setStartType(final String startType) {
        this.startType = StartType.forName(startType);
    }

    /**
     * Sets the type of application to start.
     * 
     * @param startType Applet or Application.
     */
    public final void setStartType(final StartType startType) {
        this.startType = startType;
    }
    
    /**
     * Returns the title of the frame.
     * 
     * @return Frame title.
     */
    public final String getFrameTitle() {
        return frameTitle;
    }

    /**
     * Sets the title of the frame.
     * 
     * @param frameTitle Frame title.
     */
    @Option(name = "-frameTitle", usage = "Sets the title for the frame")
    public final void setFrameTitle(final String frameTitle) {
        this.frameTitle = frameTitle;
    }
    
}
