/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.appletviewer;

import java.applet.Applet;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.UIManager;

import org.fuin.utils4swing.common.Utils4Swing;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.Option;

/**
 * Configuration for the applet viewer.
 */
public final class SimpleAppletViewerConfig {

	/** Viewer frame title. */
	private String title = "Simple Applet Viewer";

	/** Width of the viewer frame. */
	private int width = 800;

	/** Height of the viewer frame. */
	private int height = 600;

	/** Name of the applet. */
	private String name = null;

	/** Look and feel of the viewer. */
	private String lookAndFeel;

	/** Base directory for the applet. */
	private String baseDir;

	/** Command line arguments. */
	@Argument
	private List<String> arguments;

	/** Applet. */
	private Applet applet;

	/**
	 * Default constructor.
	 */
	public SimpleAppletViewerConfig() {
		super();
		this.lookAndFeel = UIManager.getSystemLookAndFeelClassName();
		this.arguments = new ArrayList<String>();
		this.baseDir = System.getProperty("user.dir");
	}

	/**
	 * Returns the height of the viewer frame.
	 * 
	 * @return Height.
	 */
	public final int getHeight() {
		return height;
	}

	/**
	 * Sets the height of the viewer frame.
	 * 
	 * @param height
	 *            Height.
	 */
	@Option(name = "-height", usage = "Sets the height of the viewer frame")
	public final void setHeight(final int height) {
		this.height = height;
	}

	/**
	 * Returns the width of the viewer frame.
	 * 
	 * @return Width.
	 */
	public final int getWidth() {
		return width;
	}

	/**
	 * Sets the width of the viewer frame.
	 * 
	 * @param width
	 *            Width.
	 */
	@Option(name = "-width", usage = "Sets the width of the viewer frame")
	public final void setWidth(final int width) {
		this.width = width;
	}

	/**
	 * Returns the name of the applet.
	 * 
	 * @return Name.
	 */
	public final String getName() {
		return name;
	}

	/**
	 * Sets the name of the applet.
	 * 
	 * @param name
	 *            Name.
	 */
	@Option(name = "-name", usage = "Sets the name of the applet")
	public final void setName(final String name) {
		this.name = name;
	}

	/**
	 * Returns the title of the viewer frame.
	 * 
	 * @return Frame title.
	 */
	public final String getTitle() {
		return title;
	}

	/**
	 * Sets the title of the viewer frame.
	 * 
	 * @param title
	 *            Frame title.
	 */
	@Option(name = "-title", usage = "Sets the title of the frame")
	public final void setTitle(final String title) {
		this.title = title;
	}

	/**
	 * Returns the name of the look and feel.
	 * 
	 * @return Full qualified class name.
	 */
	public final String getLookAndFeel() {
		return lookAndFeel;
	}

	/**
	 * Sets the name of the look and feel.
	 * 
	 * @param lookAndFeel
	 *            Full qualified class name.
	 */
	@Option(name = "-lookAndFeel", usage = "Sets the look and feel of the viewer")
	public final void setLookAndFeel(final String lookAndFeel) {
		this.lookAndFeel = lookAndFeel;
	}

	/**
	 * Returns the base directory name for the applet.
	 * 
	 * @return Base directory.
	 */
	public final String getBaseDir() {
		return baseDir;
	}

	/**
	 * Sets the base directory for the applet.
	 * 
	 * @param baseDir
	 *            Base directory.
	 */
	@Option(name = "-baseDir", usage = "Sets the base directory for the applet")
	public final void setBaseDir(final String baseDir) {
		this.baseDir = baseDir;
	}

	/**
	 * Returns command line arguments.
	 * 
	 * @return Applet arguments.
	 */
	public final List<String> getArguments() {
		return arguments;
	}

	/**
	 * Sets the rest of the command line arguments.
	 * 
	 * @param arguments
	 *            Other command line parameters than options.
	 */
	public final void setArguments(final List<String> arguments) {
		this.arguments = arguments;
	}

	/**
	 * Returns the base directory as URL. Convenience method to wrap the
	 * exceptions and directly return an URL.
	 * 
	 * @return Base directory URL.
	 */
	public final URL getBaseUrl() {
		try {
			return (new File(baseDir)).toURI().toURL();
		} catch (final MalformedURLException e) {
			throw new RuntimeException(
					"Error converting 'baseDir' into 'baseUrl'!", e);
		}
	}

	/**
	 * Returns the applet to show in the viewer.
	 * 
	 * @return Applet.
	 */
	public final Applet getApplet() {
		return applet;
	}

	/**
	 * Sets the applet to show in the viewer.
	 * 
	 * @param applet
	 *            Applet.
	 */
	public final void setApplet(final Applet applet) {
		this.applet = applet;
	}

	/**
	 * Sets the applet by creating a new instance.
	 * 
	 * @param applet
	 *            Full qualified applet class name.
	 */
	@Option(name = "-applet", usage = "Full qualified applet class name", required = true)
	public final void setApplet(final String applet) {
		this.applet = (Applet) Utils4Swing.createInstance(applet);
	}

	/**
	 * Parses the list of arguments for entries in format
	 * &lt;key&gt;=&lt;value&gt; and returns them in a map.
	 * 
	 * @return Map with parameters parsed from arguments.
	 */
	public final Map<String, String> getParams() {
		final Map<String, String> params = new HashMap<String, String>();
		for (int i = 0; i < arguments.size(); i++) {
			final String argument = arguments.get(i);
			final int p = argument.indexOf('=');
			if (p > -1) {
				final String key = argument.substring(0, p);
				final String value = argument.substring(p + 1, argument
						.length());
				params.put(key, value);
			}
		}
		return params;
	}

	/**
	 * Adds an argument to the applet.
	 * 
	 * @param arg
	 *            Parameter as &lt;key&gt;=&lt;value&gt; string.
	 */
	public final void addArgument(final String arg) {
		arguments.add(arg);
	}

	/**
	 * Adds an argument to the applet.
	 * 
	 * @param key
	 *            Key of the parameter.
	 * @param value
	 *            Value of the parameter.
	 */
	public final void addArgument(final String key, final String value) {
		arguments.add(key + "=" + value);
	}

}
