/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4swing.annotations;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.fuin.utils4swing.common.Utils4Swing;

/**
 * Helper methods for annotation.
 */
public final class AnnotationUtils {

	private AnnotationUtils() {
		throw new UnsupportedOperationException(
				"This utility class is not intended to be instanciated!");
	}

	/**
	 * Retrieve the text for a field with a <code>@Label</code> annotation.
	 * 
	 * @param clasz
	 *            Class containing the field with the <code>@Label</code>
	 *            annotation - Cannot be NULL.
	 * @param fieldName
	 *            Name of the field annotated with <code>@Label</code> - Cannot
	 *            be NULL.
	 * @param label
	 *            Label the field is annotated with - Cannot be NULL.
	 * @param locale
	 *            Locale to use - Cannot be NULL.
	 * 
	 * @return Text to display for the field.
	 */
	public static String getLabelText(final Class clasz, final String fieldName,
			final Label label, final Locale locale) {

		if (clasz == null) {
			throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
		}
		if (fieldName == null) {
			throw new IllegalArgumentException("The argument 'fieldName' cannot be null!");
		}
		if (label == null) {
			throw new IllegalArgumentException("The argument 'label' cannot be null!");
		}
		if (locale == null) {
			throw new IllegalArgumentException("The argument 'locale' cannot be null!");
		}
		try {
			final ResourceBundle bundle = getResourceBundle(clasz, locale, label);
			final String key = getKey(fieldName, label);
			return getLabelText(bundle, fieldName, label, key);
		} catch (MissingResourceException ex) {
			return fieldName;
		}
	}

	/**
	 * Try to read the text for a field that is NOT annotated with
	 * <code>@Label</code> from a resource bundle. The resource bundle is
	 * expected to be in the same package and has the same name as
	 * <code>clasz</code>. Example: For "org.fuin.example.MyClass" the resource
	 * bundle is something like "/org/fuin/example/MyClass.properties" or
	 * "/org/fuin/example/MyClass_en.properties". The <code>fieldName</code> is
	 * used as key within the resource bundle. If the key is not found
	 * <code>fieldName</code> is returned itself.
	 * 
	 * @param clasz
	 *            Class with the field - Cannot be NULL.
	 * @param fieldName
	 *            Field within the class - Cannot be NULL.
	 * @param locale
	 *            Locale to use - Cannot be NULL.
	 * 
	 * @return Text to display for the field.
	 */
	public static String getStandardText(final Class clasz, final String fieldName,
			final Locale locale) {

		if (clasz == null) {
			throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
		}
		if (fieldName == null) {
			throw new IllegalArgumentException("The argument 'fieldName' cannot be null!");
		}
		if (locale == null) {
			throw new IllegalArgumentException("The argument 'locale' cannot be null!");
		}
		try {
			final String path = Utils4Swing.getPackagePath(clasz);
			final ResourceBundle bundle = ResourceBundle.getBundle(path + "/"
					+ clasz.getSimpleName(), locale);
			final String value = bundle.getString(fieldName);
			if (value == null) {
				return fieldName;
			}
			return value;
		} catch (MissingResourceException ex) {
			return fieldName;
		}

	}

	/**
	 * Read the text for a given field from a resource bundle. If
	 * <code>key</code> is not found <code>label.value()</code> will be used
	 * instead. If <code>label.value()</code> is an empty string
	 * <code>fieldName</code> will be returned itself.
	 * 
	 * @param bundle
	 *            Resource bundle to use.
	 * @param fieldName
	 *            Name of the field.
	 * @param label
	 *            Label for the field.
	 * @param key
	 *            Key within the resource bundle.
	 * 
	 * @return Text.
	 */
	private static String getLabelText(final ResourceBundle bundle, final String fieldName,
			final Label label, final String key) {

		final String text;
		final String value = bundle.getString(key);
		if (value == null) {
			if (label.value().length() == 0) {
				text = fieldName;
			} else {
				text = label.value();
			}
		} else {
			text = value;
		}
		return text;
	}

	/**
	 * Returns the key within the resource bundle. If the
	 * <code>label.key()</code> is empty the <code>fieldName</code> is returned
	 * as key.
	 * 
	 * @param fieldName
	 *            Name of the field.
	 * @param label
	 *            Label for the field.
	 * 
	 * @return Key.
	 */
	private static String getKey(final String fieldName, final Label label) {
		final String key;
		if (label.key().length() == 0) {
			key = fieldName;
		} else {
			key = label.key();
		}
		return key;
	}

	/**
	 * Returns the resource bundle for a given class. If
	 * <code>label.bundle()</code> is empty the class name itself is used as
	 * bundle name.
	 * 
	 * @param clasz
	 *            Class to find a resource bundle for.
	 * @param locale
	 *            Locale to use.
	 * @param label
	 *            Label with bundle name.
	 * 
	 * @return Resource bundle.
	 */
	private static ResourceBundle getResourceBundle(final Class clasz, final Locale locale,
			final Label label) {
		final ResourceBundle bundle;
		if (label.bundle().length() == 0) {
			final String path = Utils4Swing.getPackagePath(clasz);
			bundle = ResourceBundle.getBundle(path + "/" + clasz.getSimpleName(), locale);
		} else {
			bundle = ResourceBundle.getBundle(label.bundle(), locale);
		}
		return bundle;
	}

}
