/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.utils4j;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Properties;

/**
 * Common utility methods for use in Java applications and libraries.
 */
public final class Utils4J {

    /**
     * Private default constructor.
     */
    private Utils4J() {
        throw new UnsupportedOperationException(
                "This utility class is not intended to be instanciated!");
    }

    /**
     * Returns the package path of a class.
     * 
     * @param clasz
     *            Class to determine the path for.
     * 
     * @return Package path for the class.
     */
    @SuppressWarnings("unchecked")
    public static String getPackagePath(final Class clasz) {
        return clasz.getPackage().getName().replace('.', '/');
    }

    /**
     * Get the path to a resource located in the same package as a given class.
     * 
     * @param clasz
     *            Class with the same package where the resource is located.
     * @param name
     *            Filename of the resource.
     * 
     * @return Resource URL.
     */
    @SuppressWarnings("unchecked")
    public static URL getResource(final Class clasz, final String name) {
        final String nameAndPath = getPackagePath(clasz) + "/" + name;
        return clasz.getResource(nameAndPath);
    }

    /**
     * Load properties from classpath.
     * 
     * @param clasz
     *            Class in the same package as the properties file - Cannot be
     *            <code>null</code>.
     * @param filename
     *            Name of the properties file (without path) - Cannot be
     *            <code>null</code>.
     * 
     * @return Properties.
     */
    @SuppressWarnings("unchecked")
    public static Properties loadProperties(final Class clasz, final String filename) {
        checkNotNull("clasz", clasz);
        checkNotNull("filename", filename);

        final String path = Utils4J.getPackagePath(clasz);
        final String resPath = "/" + path + "/" + filename;
        try {
            final Properties props = new Properties();
            final InputStream inStream = clasz.getResourceAsStream(resPath);
            if (inStream == null) {
                throw new IllegalArgumentException("Resource '" + resPath + "' was not found!");
            }
            try {
                props.load(inStream);
            } finally {
                inStream.close();
            }
            return props;
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Load properties from a file.
     * 
     * @param file
     *            Properties file - Cannot be <code>null</code>.
     * 
     * @return Properties.
     */
    public static Properties loadProperties(final File file) {
        checkNotNull("file", file);
        checkValidFile(file);
        try {
            final Properties props = new Properties();
            final InputStream inStream = new FileInputStream(file);
            try {
                props.load(inStream);
            } finally {
                inStream.close();
            }
            return props;
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Check if the argument is an existing file. If the check fails an
     * <code>IllegalArgumentException</code> is thrown.
     * 
     * @param file
     *            File to check - Cannot be <code>null</code>.
     */
    public static void checkValidFile(final File file) {
        if (!file.exists()) {
            throw new IllegalArgumentException("The file '" + file + "' does not exist!");
        }
        if (!file.isFile()) {
            throw new IllegalArgumentException("The name '" + file + "' is not a file!");
        }
    }

    /**
     * Check if the argument is an existing directory. If the check fails an
     * <code>IllegalArgumentException</code> is thrown.
     * 
     * @param dir
     *            Directory to check - Cannot be <code>null</code>.
     */
    public static void checkValidDir(final File dir) {
        if (!dir.exists()) {
            throw new IllegalArgumentException("The directory '" + dir + "' does not exist!");
        }
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException("The name '" + dir + "' is not a directory!");
        }
    }

    /**
     * Save properties to a file.
     * 
     * @param file
     *            Destination file - Cannot be <code>null</code>.
     * @param props
     *            Properties to save - Cannot be <code>null</code>.
     * @param comment
     *            Comment for the file.
     */
    public static void saveProperties(final File file, final Properties props,
            final String comment) {
        checkNotNull("file", file);
        checkNotNull("props", props);

        if (!file.getParentFile().exists()) {
            throw new IllegalArgumentException("The parent directory '" + file.getParentFile()
                    + "' does not exist [file='" + file + "']!");
        }
        try {
            final OutputStream outStream = new FileOutputStream(file);
            try {
                props.store(outStream, comment);
            } finally {
                outStream.close();
            }
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Create an instance with Class.forName(..) and wrap all exceptions into
     * RuntimeExceptions.
     * 
     * @param className
     *            Full qualified class name.
     * 
     * @return New instance of the class.
     */
    @SuppressWarnings("unchecked")
    public static Object createInstance(final String className) {
        try {
            final Class clasz = Class.forName(className);
            return clasz.newInstance();
        } catch (final ClassNotFoundException e) {
            throw new RuntimeException("Unknown class!", e);
        } catch (final InstantiationException e) {
            throw new RuntimeException("Error instanciating class!", e);
        } catch (final IllegalAccessException e) {
            throw new RuntimeException("Error accessing class!", e);
        }
    }

    /**
     * Adds an URL to the classpath.
     * 
     * @param url
     *            URL to add.
     */
    public static void addToClasspath(final String url) {
        try {
            addToClasspath(new URL(url));
        } catch (final MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Checks if the array or URLs contains the given URL.
     * 
     * @param urls
     *            Array of URLs.
     * @param url
     *            URL to find.
     * 
     * @return If the URL is in the array TRUE else FALSE.
     */
    public static boolean containsURL(final URL[] urls, final URL url) {
        for (final URL element : urls) {
            if (element.equals(url)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Creates an MD5 hash from a file.
     * 
     * @param file
     *            File to create an hash for.
     * 
     * @return Hash as text.
     */
    public static String createHash(final File file) {
        try {
            final MessageDigest messageDigest = MessageDigest.getInstance("MD5");
            final FileInputStream fis = new FileInputStream(file);
            try {
                final byte[] buf = new byte[1024];
                final BufferedInputStream in = new BufferedInputStream(fis);
                int count = 0;
                while ((count = in.read(buf)) > -1) {
                    messageDigest.update(buf, 0, count);
                }
            } finally {
                fis.close();
            }
            final byte[] digest = messageDigest.digest();
            final String digestStr = new BigInteger(1, digest).toString(16);
            if (digestStr.length() == 31) {
                return "0" + digestStr;
            } else {
                return digestStr;
            }

        } catch (NoSuchAlgorithmException ex) {
            throw new RuntimeException(ex);
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Creates an URL based on a directory a relative path and a filename.
     * 
     * @param baseUrl
     *            Directory URL.
     * @param path
     *            Relative path inside the base URL.
     * @param filename
     *            Filename without path.
     * 
     * @return URL.
     */
    public static URL createUrl(final URL baseUrl, final String path, final String filename) {
        try {
            String baseUrlStr = baseUrl.toString();
            if (!baseUrlStr.endsWith("/")) {
                baseUrlStr = baseUrlStr + "/";
            }
            final String pathStr;
            if ((path == null) || (path.length() == 0)) {
                pathStr = "";
            } else {
                if (path.endsWith("/")) {
                    pathStr = path;
                } else {
                    pathStr = path + "/";
                }
            }
            return new URL(baseUrlStr + pathStr + filename);
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Returns a relative path based on a base directory. If the
     * <code>dir</code> is not inside <code>baseDir</code> an
     * <code>IllegalArgumentException</code> is thrown.
     * 
     * @param baseDir
     *            Base directory the path is relative to.
     * @param dir
     *            Directory inside the base directory.
     * 
     * @return Path of <code>dir</code> relative to <code>baseDir</code>. If
     *         both are equal an empty string is returned.
     */
    public static String getRelativePath(final File baseDir, final File dir) {
        try {
            final String base = baseDir.getCanonicalPath();
            final String path = dir.getCanonicalPath();
            if (!path.startsWith(base)) {
                throw new IllegalArgumentException("The path '" + path
                        + "' is not inside the base directory '" + base + "'!");
            }
            if (base.equals(path)) {
                return "";
            }
            return path.substring(base.length() + 1);
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Load a file from an directory.
     * 
     * @param baseUrl
     *            Directory URL.
     * @param filename
     *            Filename without path.
     * 
     * @return Properties.
     */
    public static Properties loadProperties(final URL baseUrl, final String filename) {
        try {
            final Properties props = new Properties();
            final InputStream inStream = createUrl(baseUrl, "", filename).openStream();
            try {
                props.load(inStream);
            } finally {
                inStream.close();
            }
            return props;
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Adds an URL to the classpath.
     * 
     * @param url
     *            URL to add.
     */
    public static void addToClasspath(final URL url) {
        final ClassLoader classLoader = Utils4J.class.getClassLoader();
        if (!(classLoader instanceof URLClassLoader)) {
            throw new IllegalArgumentException("Cannot add '" + url
                    + "' to classloader because it's not an URL classloader");
        }
        final URLClassLoader urlClassLoader = (URLClassLoader) classLoader;
        if (!containsURL(urlClassLoader.getURLs(), url)) {
            try {

                final Method addURL = URLClassLoader.class.getDeclaredMethod("addURL",
                        new Class[] { URL.class });
                addURL.setAccessible(true);
                addURL.invoke(urlClassLoader, new Object[] { url });
            } catch (final NoSuchMethodException e) {
                throw new RuntimeException(e);
            } catch (final IllegalArgumentException e) {
                throw new RuntimeException(e);
            } catch (final IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (final InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Checks if a variable is not <code>null</code> and throws an
     * <code>IllegalNullArgumentException</code> if this rule is violated.
     * 
     * @param name
     *            Name of the variable to be displayed in an error message.
     * @param value
     *            Value to check for <code>null</code>.
     */
    public static void checkNotNull(final String name, final Object value) {
        if (value == null) {
            throw new IllegalNullArgumentException(name);
        }
    }

    /**
     * Creates a textual representation of the method.
     * 
     * @param returnType
     *            Return type of the method - Can be <code>null</code>.
     * @param methodName
     *            Name of the method - Cannot be <code>null</code>.
     * @param argTypes
     *            The list of parameters - Can be <code>null</code>.
     * 
     * @return Textual signature of the method.
     */
    private static String getMethodSignature(final Class<?> returnType,
            final String methodName, final Class<?>... argTypes) {
        final StringBuffer sb = new StringBuffer();
        if (returnType != null) {
            sb.append(returnType.getName());
            sb.append(" ");
        }
        sb.append(methodName);
        sb.append("(");
        if (argTypes != null) {
            for (int i = 0; i < argTypes.length; i++) {
                if (i > 0) {
                    sb.append(", ");
                }
                sb.append(argTypes[i].getName());
            }
        }
        sb.append(")");
        return sb.toString();
    }

    /**
     * Calls a method with reflection and maps all errors into one exception.
     * 
     * @param obj
     *            The object the underlying method is invoked from.
     * @param methodName
     *            Name of the Method.
     * @param argTypes
     *            The list of parameters.
     * @param args
     *            Arguments the arguments used for the method call.
     * 
     * @return The result of dispatching the method represented by this object
     *         on <code>obj</code> with parameters <code>args</code>.
     * 
     * @throws InvokeMethodFailedException
     *             Invoking the method failed for some reason.
     */
    public static Object invoke(final Object obj, final String methodName,
            final Class<?>[] argTypes, final Object[] args) throws InvokeMethodFailedException {
        Class<?> returnType = null;
        try {
            final Method method = obj.getClass().getMethod(methodName, argTypes);
            returnType = method.getReturnType();
            return method.invoke(obj, args);
        } catch (final SecurityException ex) {
            throw new InvokeMethodFailedException("Security problem with '"
                    + getMethodSignature(returnType, methodName, argTypes) + "'! ["
                    + obj.getClass().getName() + "]", ex);
        } catch (final NoSuchMethodException ex) {
            throw new InvokeMethodFailedException("Method '"
                    + getMethodSignature(returnType, methodName, argTypes) + "' not found! ["
                    + obj.getClass().getName() + "]", ex);
        } catch (IllegalArgumentException ex) {
            throw new InvokeMethodFailedException("Argument problem with '"
                    + getMethodSignature(returnType, methodName, argTypes) + "'! ["
                    + obj.getClass().getName() + "]", ex);
        } catch (IllegalAccessException ex) {
            throw new InvokeMethodFailedException("Access problem with '"
                    + getMethodSignature(returnType, methodName, argTypes) + "'! ["
                    + obj.getClass().getName() + "]", ex);
        } catch (InvocationTargetException ex) {
            throw new InvokeMethodFailedException("Got an exception when calling '"
                    + getMethodSignature(returnType, methodName, argTypes) + "'! ["
                    + obj.getClass().getName() + "]", ex);
        }

    }

}
