/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.units4j;

import java.io.File;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.Set;

import org.fuin.utils4j.Utils4J;
import org.junit.Assert;
import org.junit.Test;

/**
 * Assertion tool class for checking the test coverage.
 */
public final class AssertCoverage {

    /**
     * Private default constructor.
     */
    private AssertCoverage() {
        throw new UnsupportedOperationException(
                "This utility class is not intended to be instanciated!");
    }

    /**
     * Asserts that a every class has at least one test class. It's assumed that
     * the name of the test class follows the pattern <code>XxxxxTest</code>,
     * where <code>Xxxxx</code> is the name of the class that is tested. The
     * class must contain at least one method annotated with {@link Test}.
     * 
     * @param classes
     *            Set of classes - Cannot be <code>null</code>.
     */
    public static final void assertEveryClassHasATest(final Set<Class<?>> classes) {
        Utils4J.checkNotNull("classes", classes);

        final StringBuffer sb = new StringBuffer();

        for (final Class<?> clasz : classes) {
            final String testClassName = clasz.getName() + "Test";
            try {
                final Class<?> testClass = Class.forName(testClassName);
                if (!hasTestMethod(testClass)) {
                    sb.append("\nThe test class '" + testClassName
                            + "' contains no methods annotated with @Test");
                }
            } catch (final ClassNotFoundException ex) {
                sb.append("\nNo test class found for '" + clasz.getName() + "'");
            }

        }

        if (sb.length() > 0) {
            Assert.fail(sb.toString());
        }

    }

    /**
     * Asserts that a every class in the directory (or it's sub directories) has
     * at least one test class. It's assumed that the name of the test class
     * follows the pattern <code>XxxxxTest</code>, where <code>Xxxxx</code> is
     * the name of the class that is tested. The class must contain at least one
     * method annotated with {@link Test}.
     * 
     * @param baseDir
     *            Root directory like ("src/main/java").
     */
    public static final void assertEveryClassHasATest(final File baseDir) {
        assertEveryClassHasATest(baseDir, true);
    }

    /**
     * Asserts that a every class in the directory (or it's sub directories) has
     * at least one test class. It's assumed that the name of the test class
     * follows the pattern <code>XxxxxTest</code>, where <code>Xxxxx</code> is
     * the name of the class that is tested. The class must contain at least one
     * method annotated with {@link Test}.
     * 
     * @param baseDir
     *            Root directory like ("src/main/java").
     * @param recursive
     *            Should sub directories be included?
     */
    public static final void assertEveryClassHasATest(final File baseDir, final boolean recursive) {
        Utils4J.checkNotNull("baseDir", baseDir);
        final Set<Class<?>> classes = new HashSet<Class<?>>();
        analyzeDir(classes, baseDir, baseDir, recursive);
        assertEveryClassHasATest(classes);
    }

    /**
     * Checks if a given class has at least one method annotated with
     * {@link Test}.
     * 
     * @param testClass
     *            Class to check.
     * 
     * @return If there is a test method <code>true</code> else
     *         <code>false</code>.
     */
    private static boolean hasTestMethod(final Class<?> testClass) {
        boolean found = false;
        final Method[] methods = testClass.getMethods();
        for (final Method method : methods) {
            final Annotation testAnnotation = method.getAnnotation(Test.class);
            if (testAnnotation != null) {
                found = true;
                break;
            }
        }
        return found;
    }

    /**
     * Populates a list of classes from a given java source directory. All
     * source files must have a ".class" file in the class path.
     * 
     * @param classes
     *            Set to populate.
     * @param baseDir
     *            Root directory like ("src/main/java").
     * @param srcDir
     *            A directory inside the root directory.
     * @param recursive
     *            If sub directories should be included <code>true</code> else
     *            <code>false</code>.
     */
    static void analyzeDir(final Set<Class<?>> classes, final File baseDir, final File srcDir,
            final boolean recursive) {
        final String packageName = Utils4J.getRelativePath(baseDir, srcDir).replace(
                File.separatorChar, '.');
        final File[] files = srcDir.listFiles();
        for (int i = 0; i < files.length; i++) {
            if (files[i].isDirectory()) {
                if (recursive) {
                    analyzeDir(classes, baseDir, files[i], recursive);
                }
            } else {
                if (files[i].getName().endsWith(".java")) {
                    final String name = files[i].getName();
                    final String simpleName = name.substring(0, name.length() - 5);
                    final String className = packageName + "." + simpleName;
                    try {
                        final Class<?> clasz = Class.forName(className);
                        final int modifiers = clasz.getModifiers();
                        if (!clasz.isAnnotation() && !clasz.isInterface()
                                && !Modifier.isAbstract(modifiers)) {
                            classes.add(clasz);
                        }
                    } catch (final ClassNotFoundException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            }
        }
    }

}
