/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.units4j.dependency;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.fuin.utils4j.Utils4J;
import org.objectweb.asm.ClassReader;

/**
 * Analyzes package dependencies.
 */
public final class DependencyAnalyzer {

    private final List<DependencyError> dependencyErrors;

    private final Dependencies dependencies;

    /**
     * Constructor with XML file.
     * 
     * @param dependenciesFile
     *            XML file with allowed or forbidden dependencies - Cannot be
     *            <code>null</code>.
     * 
     * @throws InvalidDependenciesDefinitionException
     *             Formal correct (XML) but invalid dependency definition.
     */
    public DependencyAnalyzer(final File dependenciesFile)
            throws InvalidDependenciesDefinitionException {
        this(Utils.load(dependenciesFile));
    }

    /**
     * Constructor with dependency definition.
     * 
     * @param dependencies
     *            Definition of allowed or forbidden dependencies - Cannot be
     *            <code>null</code>.
     * 
     * @throws InvalidDependenciesDefinitionException
     *             Formal correct (XML) but invalid dependency definition.
     */
    public DependencyAnalyzer(final Dependencies dependencies)
            throws InvalidDependenciesDefinitionException {
        super();
        Utils4J.checkNotNull("dependencies", dependencies);
        this.dependencies = dependencies;
        this.dependencies.validate();
        dependencyErrors = new ArrayList<DependencyError>();
    }
    
    private void analyzeDir(final File classesDir) throws IOException {
        final File[] files = classesDir.listFiles();
        for (int i = 0; i < files.length; i++) {
            if (files[i].isDirectory()) {
                analyzeDir(files[i]);
            } else {
                if (files[i].getName().endsWith(".class")) {
                    analyzeClass(files[i]);
                }
            }
        }
    }

    private List<DependencyError> checkIfContainsAll(final String className,
            final Package<DependsOn> pkgFound, final Set<String> found) {
        final List<DependsOn> allowed = pkgFound.getDependencies();
        final List<DependencyError> errors = new ArrayList<DependencyError>();
        final Iterator<String> it = found.iterator();
        while (it.hasNext()) {
            final String pkgName = it.next().replace('/', '.');
            if (!pkgName.equals(pkgFound.getName()) && !dependencies.isAlwaysAllowed(pkgName)) {
                final DependsOn dep = Utils.findByName(allowed, pkgName);
                if (dep == null) {
                    errors.add(new DependencyError(className, pkgName, pkgFound.getComment()));
                }
            }
        }
        return errors;
    }

    private List<DependencyError> checkIfNotContainsAny(final String className,
            final Package<NotDependsOn> pkgFound, final Set<String> found) {
        final List<NotDependsOn> forbidden = pkgFound.getDependencies();
        final List<DependencyError> errors = new ArrayList<DependencyError>();
        final Iterator<String> it = found.iterator();
        while (it.hasNext()) {
            final String pkgName = it.next().replace('/', '.');
            final NotDependsOn ndo = Utils.findByName(dependencies.getAlwaysForbidden(), pkgName);
            if (ndo != null) {
                errors.add(new DependencyError(className, pkgName, ndo.getComment()));
            } else {
                final NotDependsOn dep = Utils.findByName(forbidden, pkgName);
                if (dep != null) {
                    final String comment;
                    if (dep.getComment() == null) {
                        comment = pkgFound.getComment();
                    } else {
                        comment = dep.getComment();
                    }
                    errors.add(new DependencyError(className, pkgName, comment));
                }
            }
        }
        return errors;
    }

    private String nameOnly(final String filename) {
        final int p = filename.lastIndexOf('.');
        if (p == -1) {
            return filename;
        }
        return filename.substring(0, p);
    }

    private void analyzeClass(final File classFile) throws IOException {
        final InputStream in = new BufferedInputStream(new FileInputStream(classFile));
        try {
            final DependencyVisitor visitor = new DependencyVisitor();
            new ClassReader(in).accept(visitor, 0);
            final Map<String, Map<String, Integer>> globals = visitor.getGlobals();
            final Set<String> jarPackages = globals.keySet();
            final String pkgName = jarPackages.iterator().next().replace('/', '.');
            final String className = pkgName + "." + nameOnly(classFile.getName());
            int idx = dependencies.getAllowed().indexOf(new Package<DependsOn>(pkgName));
            if (idx > -1) {
                final Package<DependsOn> pkgFound = dependencies.getAllowed().get(idx);
                final List<DependencyError> errors = checkIfContainsAll(className, pkgFound,
                        visitor.getPackages());
                dependencyErrors.addAll(errors);
            } else {
                idx = dependencies.getForbidden().indexOf(new Package<NotDependsOn>(pkgName));
                if (idx > -1) {
                    final Package<NotDependsOn> pkgFound = dependencies.getForbidden().get(idx);
                    final List<DependencyError> errors = checkIfNotContainsAny(className, pkgFound,
                            visitor.getPackages());
                    dependencyErrors.addAll(errors);
                }
            }
        } finally {
            in.close();
        }
    }

    /**
     * Analyze the dependencies for all classes in the directory and it's sub
     * directories.
     * 
     * @param classesDir
     *            Directory where the "*.class" files are located (something
     *            like "bin" or "classes").
     * 
     * @throws IOException
     *             Error reading the classes.
     */
    public final void analyze(final File classesDir) throws IOException {
        dependencyErrors.clear();
        analyzeDir(classesDir);
    }

    /**
     * Returns the list of dependency errors from last call to
     * {@link #analyze(File)}.
     * 
     * @return List of errors - Always non-<code>null</code> but may be empty.
     */
    public final List<DependencyError> getDependencyErrors() {
        return dependencyErrors;
    }

}
