/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.units4j.dependency;

import java.util.ArrayList;
import java.util.List;

/**
 * Package dependency rules. Caution: A package cannot be in both lists!
 */
public final class Dependencies {

    private final List<DependsOn> alwaysAllowed;

    private final List<NotDependsOn> alwaysForbidden;
    
    private final List<Package<DependsOn>> allowed;

    private final List<Package<NotDependsOn>> forbidden;

    /**
     * Default constructor.
     */
    public Dependencies() {
        super();
        this.alwaysAllowed = new ArrayList<DependsOn>();
        this.alwaysForbidden = new ArrayList<NotDependsOn>();
        this.allowed = new ArrayList<Package<DependsOn>>();
        this.forbidden = new ArrayList<Package<NotDependsOn>>();
    }

    /**
     * Returns a list of packages like "java.lang" that are always Ok to depend on.
     * 
     * @return List of packages.
     */
    public final List<DependsOn> getAlwaysAllowed() {
        return alwaysAllowed;
    }

    /**
     * Returns a list of packages that are always forbidden to depend on.
     * 
     * @return List of packages.
     */
    public final List<NotDependsOn> getAlwaysForbidden() {
        return alwaysForbidden;
    }
    
    /**
     * Checks if the package is always OK.
     * 
     * @param packageName
     *            Name of the package to check.
     * 
     * @return If the package is always allowed <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean isAlwaysAllowed(final String packageName) {
        return Utils.findByName(alwaysAllowed, packageName) != null;
    }

    
    /**
     * Checks if the package is always forbidden.
     * 
     * @param packageName
     *            Name of the package to check.
     * 
     * @return If the package is always forbidden <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean isAlwaysForbidden(final String packageName) {
        return Utils.findByName(alwaysForbidden, packageName) != null;
    }
    
    /**
     * Returns the list of allowed package dependencies.
     * 
     * @return List of explicit allowed dependencies - All other dependencies
     *         are considered to be an error.
     */
    public final List<Package<DependsOn>> getAllowed() {
        return allowed;
    }

    /**
     * Returns the list of forbidden package dependencies.
     * 
     * @return List of explicit forbidden dependencies - All other dependencies
     *         are considered to be valid.
     */
    public final List<Package<NotDependsOn>> getForbidden() {
        return forbidden;
    }

    /**
     * Checks if the definition is valid - Especially if no package is in both
     * lists.
     * 
     * @throws InvalidDependenciesDefinitionException
     *             This instance is invalid.
     */
    public final void validate() throws InvalidDependenciesDefinitionException {
        int errorCount = 0;
        final StringBuffer sb = new StringBuffer(
                "Duplicate package entries in 'allowed' and 'forbidden': ");
        for (int i = 0; i < forbidden.size(); i++) {
            final String name = forbidden.get(i).getName();
            final Package<DependsOn> dep = new Package<DependsOn>(name);
            if (allowed.indexOf(dep) > -1) {
                if (errorCount > 0) {
                    sb.append(", ");
                }
                sb.append(name);
                errorCount++;
            }
        }
        if (errorCount > 0) {
            throw new InvalidDependenciesDefinitionException(this, sb.toString());
        }
    }

}
