/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.units4j;

import java.io.File;
import java.io.IOException;
import java.util.List;

import junit.framework.Assert;

import org.fuin.units4j.dependency.Dependencies;
import org.fuin.units4j.dependency.DependencyAnalyzer;
import org.fuin.units4j.dependency.DependencyError;
import org.fuin.units4j.dependency.InvalidDependenciesDefinitionException;
import org.fuin.utils4j.Utils4J;

/**
 * Assertion tool class for checking dependencies.
 */
public final class AssertDependencies {

    /**
     * Private default constructor.
     */
    private AssertDependencies() {
        throw new UnsupportedOperationException(
                "This utility class is not intended to be instanciated!");
    }

    private static void assertIntern(final File classesDir, final DependencyAnalyzer analyzer) {
        try {
            analyzer.analyze(classesDir);
            final List<DependencyError> dependencyErrors = analyzer.getDependencyErrors();
            if (dependencyErrors.size() > 0) {
                final StringBuffer sb = new StringBuffer();
                for (int i = 0; i < dependencyErrors.size(); i++) {
                    sb.append(dependencyErrors.get(i));
                    sb.append("\n");
                }
                Assert.fail(sb.toString());
            }
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Asserts that a set of dependency rules is kept.
     * 
     * @param dependencies
     *            Definition of allowed or forbidden dependencies - Cannot be
     *            <code>null</code>.
     * @param classesDir
     *            Directory with the ".class" files to check - Cannot be
     *            <code>null</code> and must be a valid directory.
     */
    public static final void assertRules(final Dependencies dependencies, final File classesDir) {
        Utils4J.checkNotNull("dependencies", dependencies);
        Utils4J.checkNotNull("classesDir", classesDir);
        Utils4J.checkValidDir(classesDir);
        try {
            final DependencyAnalyzer analyzer = new DependencyAnalyzer(dependencies);
            assertIntern(classesDir, analyzer);
        } catch (final InvalidDependenciesDefinitionException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Asserts that a set of dependency rules is kept.
     * 
     * @param file
     *            The XML rules file - Cannot be <code>null</code> and must a a
     *            valid file.
     * @param classesDir
     *            Directory with the ".class" files to check - Cannot be
     *            <code>null</code> and must be a valid directory.
     */
    public static final void assertRules(final File file, final File classesDir) {
        Utils4J.checkNotNull("file", file);
        Utils4J.checkNotNull("classesDir", classesDir);
        Utils4J.checkValidFile(file);
        Utils4J.checkValidDir(classesDir);
        try {
            final DependencyAnalyzer analyzer = new DependencyAnalyzer(file);
            assertIntern(classesDir, analyzer);
        } catch (final InvalidDependenciesDefinitionException ex) {
            throw new RuntimeException(ex);
        }
    }

}
