/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist.factory;

import java.lang.reflect.Method;
import java.util.List;

import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgMethod;

/**
 * Creates an implementation of an interface.
 */
public class ImplementationFactory {

	private final SgClassPool pool;

	/**
	 * Constructor with class pool.
	 * 
	 * @param pool
	 *            Class pool.
	 */
	public ImplementationFactory(final SgClassPool pool) {
		super();

		assureNotNull("pool", pool);
		this.pool = pool;

	}

	/**
	 * Creates an implementation of the interface.
	 * 
	 * @param implPackageName
	 *            Name of the implementation package - Cannot be null.
	 * @param implClassName
	 *            Name of the implementation class - Cannot be null.
	 * @param listener
	 *            Creates the bodies for all methods - Cannot be null.
	 * @param intf
	 *            One or more interfaces.
	 * 
	 * @return New object implementing the interface.
	 */
	public final SgClass create(final String implPackageName, final String implClassName,
			final ImplementationFactoryListener listener, final Class<?>... intf) {
		return create(implPackageName, implClassName, null, null, listener, intf);
	}

	/**
	 * Creates an implementation of the interface.
	 * 
	 * @param implPackageName
	 *            Name of the implementation package - Cannot be null.
	 * @param implClassName
	 *            Name of the implementation class - Cannot be null.
	 * @param superClass
	 *            Parent class or <code>null</code>.
	 * @param enclosingClass
	 *            Outer class or <code>null</code>.
	 * @param listener
	 *            Creates the bodies for all methods - Cannot be null.
	 * @param intf
	 *            One or more interfaces.
	 * 
	 * @return New object implementing the interface.
	 */
	public final SgClass create(final String implPackageName, final String implClassName,
			final SgClass superClass, final SgClass enclosingClass,
			final ImplementationFactoryListener listener, final Class<?>... intf) {

		assureNotNull("implPackageName", implPackageName);
		assureNotNull("implClassName", implClassName);
		assureNotNull("listener", listener);
		assureNotNull("intf", intf);
		assureNotEmpty("intf", intf);
		assureAllInterfaces(intf);

		// Create class with all interfaces
		final SgClass clasz = new SgClass("public", implPackageName, implClassName, superClass,
				false, enclosingClass);
		for (int i = 0; i < intf.length; i++) {
			clasz.addInterface(SgClass.create(pool, intf[i]));
		}
		listener.afterClassCreated(clasz);

		// Iterate through interfaces and add methods
		for (int i = 0; i < intf.length; i++) {
			addInterfaceMethods(clasz, intf[i], listener);
		}

		return clasz;
	}

	private void addInterfaceMethods(final SgClass clasz, final Class<?> intf,
			final ImplementationFactoryListener listener) {

		final Method[] methods = intf.getMethods();
		for (int j = 0; j < methods.length; j++) {

			// Create method
			final String name = methods[j].getName();
			final SgClass returnType;
			if (methods[j].getReturnType() == null) {
				returnType = SgClass.VOID;
			} else {
				returnType = SgClass.create(pool, methods[j].getReturnType());
			}
			final SgMethod method = new SgMethod(clasz, "public", returnType, name);

			// Add arguments
			final Class<?>[] paramTypes = methods[j].getParameterTypes();
			for (int k = 0; k < paramTypes.length; k++) {
				final SgClass paramType = SgClass.create(pool, paramTypes[k]);
				method.addArgument(new SgArgument(method, paramType, ("arg" + k)));
			}

			// Add exceptions
			final Class<?>[] exceptionTypes = methods[j].getExceptionTypes();
			for (int k = 0; k < exceptionTypes.length; k++) {
				method.addException(SgClass.create(pool, exceptionTypes[k]));
			}

			// Add source
			final List<String> lines = listener.createBody(method);
			for (int k = 0; k < lines.size(); k++) {
				method.addBodyLine(lines.get(k));
			}

		}

	}

	private void assureNotNull(final String name, final Object value) {
		if (value == null) {
			throw new IllegalArgumentException("The argument '" + name + "' cannot be null!");
		}
	}

	private void assureNotEmpty(final String name, final Object[] value) {
		if (value.length == 0) {
			throw new IllegalArgumentException("The argument '" + name
					+ "' cannot be an empty array!");
		}
	}

	private void assureAllInterfaces(final Class<?>... intf) {
		for (int i = 0; i < intf.length; i++) {
			if (!intf[i].isInterface()) {
				throw new IllegalArgumentException("Expected an interface: " + intf[i].getName()
						+ " [" + i + "]");
			}
		}
	}

}
