/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.ArrayList;
import java.util.List;

/**
 * Helper class to make writing source lines a bit easier.
 */
public final class SourceWriter {

    private final List<String> lines = new ArrayList<String>();

    private StringBuffer line = new StringBuffer();

    /**
     * Default constructor.
     */
    public SourceWriter() {
        super();
    }

    /**
     * Write a string into the current line.
     * 
     * @param str
     *            String to add.
     */
    public final void write(final String str) {
        if (str == null) {
            return;
        }
        line.append(str);
    }

    /**
     * Append the string to the current line and start a new one.
     * 
     * @param str
     *            String to add.
     */
    public final void writeLn(final String str) {
        if (str != null) {
            line.append(str);
        }
        lines.add(line.toString());
        line = new StringBuffer();
    }

    /**
     * Start a new line.
     */
    public final void writeLn() {
        lines.add(line.toString());
        line = new StringBuffer();
    }

    /**
     * Write a string to the current line. If the line is not empty a divider
     * will be added before the string.
     * 
     * @param divider
     *            Divider to use for non-empty lines.
     * @param str
     *            String to add.
     */
    public final void write(final String divider, final String str) {
        if (line.length() > 0) {
            line.append(divider);
        }
        write(str);
    }

    /**
     * Write a string to the current line. If the line is not empty a divider
     * will be added before the string. After adding the string a new line will
     * be started.
     * 
     * @param divider
     *            Divider to use for non-empty lines.
     * @param str
     *            String to add.
     */
    public final void writeLn(final String divider, final String str) {
        if (line.length() > 0) {
            line.append(divider);
        }
        writeLn(str);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String toString() {
        lines.add(line.toString());
        line = new StringBuffer();
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < lines.size(); i++) {
            sb.append(lines.get(i));
            if (i < lines.size() - 1) {
                sb.append("\n");
            }
        }
        return sb.toString();
    }

}
