/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.List;

import javassist.Modifier;

/**
 * Some helper for the package.
 */
public final class SgUtils {

    private SgUtils() {
        throw new UnsupportedOperationException(
                "It's not allowed to create an instance of this class!");
    }

    private static void writeVisibility(final SourceWriter src, final int modifiers) {
        if (Modifier.isPrivate(modifiers)) {
            src.write(" ", "private");
        }
        if (Modifier.isProtected(modifiers)) {
            src.write(" ", "protected");
        }
        if (Modifier.isPublic(modifiers)) {
            src.write(" ", "public");
        }
    }

    /**
     * Create modifier text for classes.
     * 
     * @param modifiers
     *            Modifiers.
     * @param isInterface
     *            Are the modifiers from an interface?
     * @param isInnerClass
     *            Is it an inner class?
     * 
     * @return Space separated modifier names.
     */
    public static String createClassModifiers(final int modifiers, final boolean isInterface,
            final boolean isInnerClass) {
        final SourceWriter src = new SourceWriter();
        writeVisibility(src, modifiers);
        if ((!(isInterface && isInnerClass)) && Modifier.isStatic(modifiers)) {
            src.write(" ", "static");
        }
        if ((!isInterface) && Modifier.isAbstract(modifiers)) {
            src.write(" ", "abstract");
        }
        if (Modifier.isFinal(modifiers)) {
            src.write(" ", "final");
        }
        return src.toString();
    }

    /**
     * Create modifier text for fields.
     * 
     * @param modifiers
     *            Modifiers.
     * 
     * @return Space separated modifier names.
     */
    public static String createFieldModifiers(final int modifiers) {
        final SourceWriter src = new SourceWriter();
        writeVisibility(src, modifiers);
        if (Modifier.isStatic(modifiers)) {
            src.write(" ", "static");
        }
        if (Modifier.isFinal(modifiers)) {
            src.write(" ", "final");
        }
        if (Modifier.isTransient(modifiers)) {
            src.write(" ", "transient");
        }
        if (Modifier.isVolatile(modifiers)) {
            src.write(" ", "volatile");
        }
        return src.toString();
    }

    /**
     * Create modifier text for methods.
     * 
     * @param modifiers
     *            Modifiers.
     * @param isInterface
     *            Are the modifiers from an interface?
     * 
     * @return Space separated modifier names.
     */
    public static String createMethodModifiers(final int modifiers, final boolean isInterface) {
        final SourceWriter src = new SourceWriter();
        writeVisibility(src, modifiers);
        if (Modifier.isStatic(modifiers)) {
            src.write(" ", "static");
        }
        if ((!isInterface) && Modifier.isAbstract(modifiers)) {
            src.write(" ", "abstract");
        }
        if (Modifier.isFinal(modifiers)) {
            src.write(" ", "final");
        }
        if (Modifier.isSynchronized(modifiers)) {
            src.write(" ", "synchronized");
        }
        return src.toString();
    }

    /**
     * Create modifier text for constructors.
     * 
     * @param modifiers
     *            Modifiers.
     * 
     * @return Space separated modifier names.
     */
    public static String createConstructorModifiers(final int modifiers) {
        final SourceWriter src = new SourceWriter();
        writeVisibility(src, modifiers);
        return src.toString();
    }

    /**
     * Inserts an underscore before every upper case character and returns an
     * all lower case string. If the first character is upper case an underscore
     * will not be inserted.
     * 
     * @param str
     *            String to convert.
     * 
     * @return Lower case + underscored text.
     */
    public static String uppercaseToUnderscore(final String str) {
        if (str == null) {
            return null;
        }
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < str.length(); i++) {
            final char ch = str.charAt(i);
            if (Character.isUpperCase(ch)) {
                if (i > 0) {
                    sb.append("_");
                }
                sb.append(Character.toLowerCase(ch));
            } else {
                sb.append(ch);
            }
        }
        return sb.toString();
    }

    /**
     * Converts the first character into upper case.
     * 
     * @param str
     *            String to convert.
     * 
     * @return Same string but first character upper case.
     */
    public static String firstCharUpper(final String str) {
        if (str == null) {
            return null;
        }
        if (str.length() == 0) {
            return str;
        }
        if (str.length() == 1) {
            return "" + Character.toUpperCase(str.charAt(0));
        }
        return Character.toUpperCase(str.charAt(0)) + str.substring(1);
    }

    /**
     * Merge two packages into one. If any package is null or empty no "." will
     * be added. If both packages are null an empty string will be returned.
     * 
     * @param package1
     *            First package - Can also be null or empty.
     * @param package2
     *            Second package - Can also be null or empty.
     * 
     * @return Both packages added with ".".
     */
    public static String concatPackages(final String package1, final String package2) {
        if ((package1 == null) || (package1.length() == 0)) {
            if ((package2 == null) || (package2.length() == 0)) {
                return "";
            } else {
                return package2;
            }
        }
        return package1 + "." + package2;
    }

    /**
     * Creates an <code>toString()</code> method with all fields.
     * 
     * @param pool
     *            Pool to use.
     * @param clasz
     *            Class to add the new method to.
     * @param fields
     *            List of fields to output.
     */
    public static void addToStringMethod(final SgClassPool pool, final SgClass clasz,
            final List<SgField> fields) {
        final SgMethod m = new SgMethod(clasz, "public", SgClass.create(pool,
                String.class), "toString");
        m.addBodyLine("return getClass().getSimpleName() + \"{\"");
        for (int i = 0; i < fields.size(); i++) {
            final SgField field = fields.get(i);
            final String nameValue = " + \"" + field.getName() + "=\" + " + field.getName();
            if (i < fields.size() - 1) {
                m.addBodyLine(nameValue + " + \", \"");
            } else {
                m.addBodyLine(nameValue);
            }
        }
        m.addBodyLine(" + \"}\";");
        clasz.addMethod(m);
    }

}
