/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.ArrayList;
import java.util.List;

/**
 * A method.
 */
public final class SgMethod extends SgBehavior {

    private final SgClass returnType;

    private final String name;

    private final List<String> body;

    private final List<SgAnnotation> annotations;

    /**
     * Constructor.
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param modifiers
     *            Modifiers for the constructor/method - Cannot be null (but
     *            empty).
     * @param returnType
     *            Return type of the method - Cannot be null (Use VOID in model
     *            class for no return value).
     * @param name
     *            Name of the method.
     */
    public SgMethod(final SgClass clasz, final String modifiers,
            final SgClass returnType, final String name) {
        super(clasz, modifiers);
        if (returnType == null) {
            throw new IllegalArgumentException("The argument 'returnType' cannot be NULL!");
        }
        this.returnType = returnType;

        if (name == null) {
            throw new IllegalArgumentException("The argument 'name' cannot be NULL!");
        }
        this.name = name;

        body = new ArrayList<String>();
        annotations = new ArrayList<SgAnnotation>();

        // TODO Check if the class not already contains a method with the same
        // name and arguments!
    }

    /**
     * Return the return type of the method.
     * 
     * @return Type - Always non-null.
     */
    public final SgClass getReturnType() {
        return returnType;
    }

    /**
     * Returns the body of the method.
     * 
     * @return Body - Always non-null (but maybe empty).
     */
    public final List<String> getBody() {
        return body;
    }

    /**
     * Add a new line to the body.
     * 
     * @param line
     *            Line to add - Cannot be null (but empty).
     */
    public final void addBodyLine(final String line) {
        if (line == null) {
            throw new IllegalArgumentException("The argument 'line' cannot be NULL!");
        }
        body.add(line.trim());
    }

    /**
     * Returns the annotations for this method.
     * 
     * @return List of annotations - Always non-null.
     */
    public final List<SgAnnotation> getAnnotations() {
        return annotations;
    }

    /**
     * Adds an annotation.
     * 
     * @param annotation
     *            Annotation to add - Cannot be null.
     */
    public final void addAnnotation(final SgAnnotation annotation) {
        if (annotation == null) {
            throw new IllegalArgumentException("The argument 'annotation' cannot be NULL!");
        }
        annotations.add(annotation);
    }

    /**
     * Adds a list of annotations. The internal list will not be cleared! The
     * annotations will simply be added with <code>addAll(..)</code>.
     * 
     * @param annotations
     *            Annotations to add - Cannot be null.
     */
    public final void addAnnotations(final List<SgAnnotation> annotations) {
        if (annotations == null) {
            throw new IllegalArgumentException("The argument 'annotations' cannot be NULL!");
        }
        this.annotations.addAll(annotations);
    }

    /**
     * Checks if a given annotation is in the list.
     * 
     * @param name
     *            Name of the annotation to find - Cannot be null.
     * 
     * @return If it's found true else false.
     */
    public final boolean hasAnnotation(final String name) {
        if (name == null) {
            throw new IllegalArgumentException("The argument 'name' cannot be NULL!");
        }
        for (int i = 0; i < annotations.size(); i++) {
            final SgAnnotation annotation = annotations.get(i);
            if (annotation.getName().equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the name of the method.
     * 
     * @return Name - Always non-null.
     */
    public final String getName() {
        return name;
    }

    /**
     * Returns the name of the method with an "underscore" inserted before all
     * upper case characters and all characters converted to lower case.
     * 
     * @return Name usable as a package - Always non-null.
     */
    public final String getNameAsPackage() {
        return SgUtils.uppercaseToUnderscore(getName());
    }

    /**
     * Returns the "signature" of the method.
     * 
     * @return Modifiers, return type, name and arguments - Always non-null.
     */
    public final String getSignature() {
        final StringBuffer sb = new StringBuffer();
        if (getModifiers().length() > 0) {
            sb.append(getModifiers());
            sb.append(" ");
        }
        sb.append(returnType.getName());
        sb.append(" ");
        sb.append(getName());
        sb.append("(");
        for (int i = 0; i < getArguments().size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(getArguments().get(i));
        }
        sb.append(")");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String toString() {
        final StringBuffer sb = new StringBuffer();
        if (annotations.size() > 0) {
            for (int i = 0; i < annotations.size(); i++) {
                if (i > 0) {
                    sb.append(" ");
                }
                sb.append(annotations.get(i));
            }
            sb.append("\n");
        }
        sb.append(getSignature());
        if (getJmsMvcClass().isInterface()) {
            sb.append(";");
        } else {
            sb.append("{\n");
            if (body.size() == 0) {
                sb.append("// No method source available\n");
            } else {
                for (int i = 0; i < body.size(); i++) {
                    sb.append(body.get(i));
                    sb.append("\n");
                }
            }
            sb.append("}\n");
        }
        return sb.toString();
    }

}
