/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.ArrayList;
import java.util.List;

/**
 * A field of a class.
 */
public final class SgField {

    private final String modifiers;

    private final SgClass clasz;

    private final String name;

    private String initializer = null;

    private final List<SgAnnotation> annotations;

    /**
     * Constructor with modifiers, class and name.
     * 
     * @param modifiers
     *            Modifiers (separated by space) - Cannot be null (but empty).
     * @param clasz
     *            Class the field belongs to - Cannot be null.
     * @param name
     *            Name of the field - Cannot be null.
     */
    public SgField(final String modifiers, final SgClass clasz, final String name) {
        super();
        if (modifiers == null) {
            throw new IllegalArgumentException("The argument 'modifiers' cannot be null!");
        }
        this.modifiers = modifiers;

        if (clasz == null) {
            throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
        }
        this.clasz = clasz;

        if (name == null) {
            throw new IllegalArgumentException("The argument 'name' cannot be null!");
        }
        this.name = name;

        // TODO Does not work when analyzing classes... Check why!
//        if (clasz.findFieldByName(name) != null) {
//            throw new IllegalArgumentException("The class '" + clasz.getName()
//                    + "' already contains a field with name '" + name + "'!");
//        }

        annotations = new ArrayList<SgAnnotation>();
    }

    /**
     * Returns the modifiers.
     * 
     * @return Modifiers (separated by space) - Always non-null.
     */
    public final String getModifiers() {
        return modifiers;
    }

    /**
     * Returns the class.
     * 
     * @return Class the field belongs to - Always non-null.
     */
    public final SgClass getClasz() {
        return clasz;
    }

    /**
     * Returns the name.
     * 
     * @return Name of the field - Always non-null.
     */
    public final String getName() {
        return name;
    }

    /**
     * Returns the initializer.
     * 
     * @return Initializer for the field - Maybe null.
     */
    public final String getInitializer() {
        return initializer;
    }

    /**
     * Sets the initializer to a new value.
     * 
     * @param initializer
     *            Initializer for the field - Can be null.
     */
    public final void setInitializer(final String initializer) {
        this.initializer = initializer;
    }

    /**
     * Returns the annotations for this field.
     * 
     * @return List of annotations - Always non-null.
     */
    public final List<SgAnnotation> getAnnotations() {
        return annotations;
    }

    /**
     * Adds an annotation.
     * 
     * @param annotation
     *            Annotation to add - Cannot be null.
     */
    public final void addAnnotation(final SgAnnotation annotation) {
        if (annotation == null) {
            throw new IllegalArgumentException("The argument 'annotation' cannot be NULL!");
        }
        annotations.add(annotation);
    }

    /**
     * Adds a list of annotations. The internal list will not be cleared! The
     * annotations will simply be added with <code>addAll(..)</code>.
     * 
     * @param annotations
     *            Annotations to add - Cannot be null.
     */
    public final void addAnnotations(final List<SgAnnotation> annotations) {
        if (annotations == null) {
            throw new IllegalArgumentException("The argument 'annotations' cannot be NULL!");
        }
        this.annotations.addAll(annotations);
    }

    /**
     * Checks if a given annotation is in the list.
     * 
     * @param name
     *            Name of the annotation to find - Cannot be null.
     * 
     * @return If it's found true else false.
     */
    public final boolean hasAnnotation(final String name) {
        if (name == null) {
            throw new IllegalArgumentException("The argument 'name' cannot be NULL!");
        }
        for (int i = 0; i < annotations.size(); i++) {
            final SgAnnotation annotation = annotations.get(i);
            if (annotation.getName().equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String toString() {
        final StringBuffer sb = new StringBuffer();
        if (annotations.size() > 0) {
            for (int i = 0; i < annotations.size(); i++) {
                if (i > 0) {
                    sb.append(" ");
                }
                sb.append(annotations.get(i));
            }
            sb.append("\n");
        }
        if (modifiers.length() == 0) {
            sb.append(clasz.getSourceName() + " " + name);
        } else {
            sb.append(modifiers + " " + clasz.getSourceName() + " " + name);
        }
        if (initializer == null) {
            sb.append(" /** No initializer source available */ ");
        } else {
            if (initializer.length() > 0) {
                sb.append(" = ");
                sb.append(initializer);
            }
        }
        sb.append(";\n");
        return sb.toString();
    }

}
