/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.ArrayList;
import java.util.List;

/**
 * A constructor.
 */
public final class SgConstructor extends SgBehavior {

    private final List<String> body;

    /**
     * Constructor with class. Modifier defaults to "public".
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     */
    public SgConstructor(final SgClass clasz) {
        this(clasz, "public");
    }

    /**
     * Constructor with class and modifiers.
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param modifiers
     *            Modifiers for the constructor/method - Cannot be null (but
     *            empty).
     */
    public SgConstructor(final SgClass clasz, final String modifiers) {
        this(clasz, modifiers, new ArrayList<SgArgument>());
    }

    /**
     * Constructor with class and arguments. Modifier defaults to "public".
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param arguments
     *            Constructor/Method arguments to copy - Cannot be null.
     */
    public SgConstructor(final SgClass clasz, final List<SgArgument> arguments) {
        this(clasz, "public", arguments);
    }

    /**
     * Constructor with class and arguments.
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param modifiers
     *            Modifiers for the constructor/method - Cannot be null (but
     *            empty).
     * @param arguments
     *            Constructor/Method arguments to copy - Cannot be null.
     */
    public SgConstructor(final SgClass clasz, final String modifiers,
            final List<SgArgument> arguments) {
        super(clasz, modifiers, arguments);
        body = new ArrayList<String>();
        // TODO Check if the class not already contains a constructor with the
        // same name and arguments!
    }

    /**
     * Returns the body of the constructor.
     * 
     * @return Body - Always non-null but maybe empty.
     */
    public final List<String> getBody() {
        return body;
    }

    /**
     * Add a new line to the body.
     * 
     * @param line
     *            Line to add - Cannot be null (but empty).
     */
    public final void addBodyLine(final String line) {
        if (line == null) {
            throw new IllegalArgumentException("The argument 'line' cannot be null!");
        }
        body.add(line);
    }

    /**
     * Returns the "signature" of the constructor.
     * 
     * @return Modifiers and arguments.
     */
    public final String getSignature() {
        final StringBuffer sb = new StringBuffer();
        if (getModifiers().length() > 0) {
            sb.append(getModifiers());
            sb.append(" ");
        }
        sb.append(getJmsMvcClass().getSimpleName());
        sb.append("(");
        for (int i = 0; i < getArguments().size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(getArguments().get(i));
        }
        sb.append(")");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String toString() {
        final StringBuffer sb = new StringBuffer();
        sb.append(getSignature());
        if (getJmsMvcClass().isInterface()) {
            sb.append(";");
        } else {
            sb.append("{\n");
            if (body.size() == 0) {
                sb.append("// No method source available\n");
            } else {
                for (int i = 0; i < body.size(); i++) {
                    sb.append(body.get(i));
                    sb.append("\n");
                }
            }
            sb.append("}\n");
        }
        return sb.toString();
    }

}
