/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.util.ArrayList;
import java.util.List;

/**
 * Base class for constructors and methods.
 */
public abstract class SgBehavior {

    private final SgClass clasz;

    private final String modifiers;

    private final List<SgArgument> arguments;

    private final List<SgClass> exceptions;

    /**
     * Constructor with class and modifiers.
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param modifiers
     *            Modifiers for the constructor/method - Cannot be null (but
     *            empty).
     */
    public SgBehavior(final SgClass clasz, final String modifiers) {
        super();
        if (clasz == null) {
            throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
        }
        this.clasz = clasz;

        if (modifiers == null) {
            throw new IllegalArgumentException("The argument 'modifiers' cannot be null!");
        }
        this.modifiers = modifiers;

        this.arguments = new ArrayList<SgArgument>();
        this.exceptions = new ArrayList<SgClass>();
    }

    /**
     * Constructor with arguments.
     * 
     * @param clasz
     *            Class the behavior belongs to - Cannot be null.
     * @param modifiers
     *            Modifiers for the constructor/method (space separated)- Cannot
     *            be null (but empty).
     * @param arguments
     *            Constructor/Method arguments to copy - Cannot be null.
     */
    public SgBehavior(final SgClass clasz, final String modifiers,
            final List<SgArgument> arguments) {
        super();
        if (clasz == null) {
            throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
        }
        this.clasz = clasz;

        if (modifiers == null) {
            throw new IllegalArgumentException("The argument 'modifiers' cannot be null!");
        }
        this.modifiers = modifiers;

        this.arguments = new ArrayList<SgArgument>();
        this.arguments.addAll(arguments);
        this.exceptions = new ArrayList<SgClass>();
    }

    /**
     * Returns the class the behavior belongs to.
     * 
     * @return Class - Always non-null.
     */
    public final SgClass getJmsMvcClass() {
        return clasz;
    }

    /**
     * Returns the modifiers for the constructor/method.
     * 
     * @return Modifiers (space separated) - Always non-null (but maybe empty).
     */
    public final String getModifiers() {
        return modifiers;
    }

    /**
     * Returns the constructor/Method arguments.
     * 
     * @return Arguments - Always non-null.
     */
    public final List<SgArgument> getArguments() {
        return arguments;
    }

    private static final String commaSeparated(final List<SgArgument> args) {
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < args.size(); i++) {
            if (i > 0) {
                sb.append(",");
            }
            sb.append(args.get(i).getName());
        }
        return sb.toString();
    }

    /**
     * Returns the argument names from 0..(N+less) separated by a comma.
     * 
     * @param less
     *            The number of arguments NOT to return from the end of the
     *            list. It's always a NEGATIVE value.
     * 
     * @return Arguments without "less" arguments at the end of the list.
     */
    public final String getCommaSeparatedArgumentNames(final int less) {
        final List<SgArgument> args = getArguments(less);
        return commaSeparated(args);
    }

    /**
     * Returns the argument names separated by a comma.
     * 
     * @return Arguments.
     */
    public final String getCommaSeparatedArgumentNames() {
        return commaSeparated(arguments);
    }

    /**
     * Returns the arguments from 0..(N+less).
     * 
     * @param less
     *            The number of arguments NOT to return from the end of the
     *            list. It's always a NEGATIVE value.
     * 
     * @return Arguments without "less" arguments at the end of the list.
     */
    public final List<SgArgument> getArguments(final int less) {
        if (less >= 0) {
            throw new IllegalArgumentException("Only negative values are allowed! [" + less
                    + "]");
        }
        final int count = arguments.size() + less;
        if (count < 0) {
            throw new IllegalArgumentException("There are only " + arguments.size()
                    + " arguments! Subtracting '" + less + "' would be below zero!");
        }
        final List<SgArgument> list = new ArrayList<SgArgument>();
        for (int i = 0; i < count; i++) {
            list.add(arguments.get(i));
        }
        return list;
    }

    /**
     * Returns the last argument of the list.
     * 
     * @return Last argument or null if the list is empty.
     */
    public final SgArgument getLastArgument() {
        final int size = arguments.size();
        if (size == 0) {
            return null;
        }
        return arguments.get(size - 1);
    }

    /**
     * Adds an argument to the list.
     * 
     * @param arg
     *            Argument to add - Non null.
     */
    public final void addArgument(final SgArgument arg) {
        if (arg == null) {
            throw new IllegalArgumentException("The argument 'arg' cannot be null!");
        }
        arguments.add(arg);
    }

    /**
     * Returns the list of exceptions thrown by the constructor/method.
     * 
     * @return List of exceptions - Always non-null.
     */
    public final List<SgClass> getExceptions() {
        return exceptions;
    }

    /**
     * Adds an exception to the list.
     * 
     * @param clasz
     *            Exception to add.
     */
    public final void addException(final SgClass clasz) {
        if (clasz == null) {
            throw new IllegalArgumentException("The argument 'clasz' cannot be null!");
        }
        // TODO Check if any superclass is of type Exception.
        exceptions.add(clasz);
    }

}
