/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

/**
 * Argument for a behavior (constructor or method).
 */
public final class SgArgument {

    private final SgClass type;

    private final String name;

    /**
     * Constructor with argument.
     * 
     * @param argument
     *            Argument to copy type and name from - Cannot be null.
     */
    public SgArgument(final SgArgument argument) {
        super();
        if (argument == null) {
            throw new IllegalArgumentException("The argument 'argument' cannot be null!");
        }
        this.type = argument.getType();
        this.name = argument.getName();
    }

    /**
     * Constructor with type and name.
     * 
     * @param type
     *            Type - Cannot be null.
     * @param name
     *            Name - Cannot be null and cannot be empty.
     */
    public SgArgument(final SgClass type, final String name) {
        super();
        if (type == null) {
            throw new IllegalArgumentException("The argument 'type' cannot be null!");
        }
        this.type = type;

        final String trimmed = name.trim();
        if (trimmed.length() == 0) {
            throw new IllegalArgumentException(
                    "The argument 'name' cannot be an empty string!");
        }
        this.name = trimmed;
    }

    /**
     * Returns the type of the argument.
     * 
     * @return Type - Always non-null.
     */
    public final SgClass getType() {
        return type;
    }

    /**
     * Returns the name of the argument.
     * 
     * @return Name - Always non-null and not empty.
     */
    public final String getName() {
        return name;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String toString() {
        return type.getSourceName() + " " + name;
    }

}
