/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.srcgen4javassist;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMethod;
import javassist.CtNewConstructor;
import javassist.CtNewMethod;
import javassist.NotFoundException;

/**
 * Creates the byte code for a model class and has some more helper methods.
 */
public final class ByteCodeGenerator {

    private ByteCodeGenerator() {
        throw new UnsupportedOperationException(
                "It's not allowed to create an instance of this class!");
    }

    /**
     * Creates a Javassist class from a given model class.
     * 
     * @param modelClass
     *            Model class to convert into a Javassist class.
     * 
     * @return Javassist class.
     * 
     * @throws NotFoundException
     *             A class or interface from the model was not found.
     * @throws CannotCompileException
     *             Some source from the model cannot be compiled.
     */
    private static CtClass createCtClass(final SgClass modelClass)
            throws NotFoundException, CannotCompileException {

        final ClassPool pool = ClassPool.getDefault();

        // Create class
        final CtClass clasz = pool.makeClass(modelClass.getName());
        // TODO Add modifiers

        // Add superclass
        if (modelClass.getSuperClass() != null) {
            clasz.setSuperclass(pool.get(modelClass.getSuperClass().getName()));
        }

        // Add interfaces
        final List<SgClass> interfaces = modelClass.getInterfaces();
        if (interfaces.size() > 0) {
            for (int i = 0; i < interfaces.size(); i++) {
                final SgClass intf = interfaces.get(i);
                clasz.addInterface(pool.get(intf.getName()));
            }
        }

        // Add fields
        final List<SgField> fields = modelClass.getFields();
        for (int i = 0; i < fields.size(); i++) {
            final SgField field = fields.get(i);
            final String src = field.toString();
            final CtField ctField = CtField.make(src, clasz);
            clasz.addField(ctField);
        }

        // Add constructors
        final List<SgConstructor> constructors = modelClass.getConstructors();
        for (int i = 0; i < constructors.size(); i++) {
            final SgConstructor constructor = constructors.get(i);
            final String src = constructor.toString();
            final CtConstructor ctConstructor = CtNewConstructor.make(src, clasz);
            clasz.addConstructor(ctConstructor);
            // TODO Add exceptions
        }

        // Add methods
        final List<SgMethod> methods = modelClass.getMethods();
        for (int i = 0; i < methods.size(); i++) {
            final SgMethod method = methods.get(i);
            final String src = method.toString();
            final CtMethod ctMethod = CtNewMethod.make(src, clasz);
            clasz.addMethod(ctMethod);
            // TODO Add exceptions
        }

        return clasz;
    }

    /**
     * Generates the byte code for a model class and returns it. The class will
     * only be created if it's not known (<code>Class.forName(..)</code> throws
     * a <code>ClassNotFoundException</code>). If it's known the class will be
     * returned instead of creating a new one.
     * 
     * @param modelClass
     *            Model class to create the byte code for.
     * 
     * @return Class.
     */
    @SuppressWarnings("unchecked")
    public static Class createClass(final SgClass modelClass) {
        // We don't want to create it if it already exists
        Class implClass = loadClass(modelClass);
        if (implClass == null) {
            try {
                // Create class
                final CtClass clasz = createCtClass(modelClass);
                implClass = clasz.toClass();
            } catch (final NotFoundException e) {
                throw new RuntimeException(e);
            } catch (final CannotCompileException e) {
                throw new RuntimeException(e);
            }
        }
        return implClass;
    }

    /**
     * Tries to load the model class calling <code>Class.forName(..)</code>.
     * 
     * @param modelClass
     *            Model class to lookup.
     * 
     * @return Class (if it already exists) or null if it's unknown.
     */
    @SuppressWarnings("unchecked")
    private static Class loadClass(final SgClass modelClass) {
        Class implClass;
        try {
            implClass = Class.forName(modelClass.getName());
            // The class already exists!
        } catch (final ClassNotFoundException e) {
            implClass = null;
        }
        return implClass;
    }

    /**
     * Creates an instance mapping all exceptions into
     * <code>RuntimeException</code>.
     * 
     * @param clasz
     *            Class to create an instance for.
     * @param argTypes
     *            Argument types of the constructor to use.
     * @param initArgs
     *            Argument values for the constructor.
     * 
     * @return New instance.
     */
    @SuppressWarnings("unchecked")
    public static Object createInstance(final Class clasz, final Class[] argTypes,
            final Object[] initArgs) {
        try {
            final Constructor constructor = clasz.getConstructor(argTypes);
            return constructor.newInstance(initArgs);
        } catch (final NoSuchMethodException ex) {
            throw new RuntimeException(ex);
        } catch (final InstantiationException ex) {
            throw new RuntimeException(ex);
        } catch (final IllegalAccessException ex) {
            throw new RuntimeException(ex);
        } catch (final InvocationTargetException ex) {
            throw new RuntimeException(ex);
        }

    }

    /**
     * Creates an instance from a model class mapping all exceptions into
     * <code>RuntimeException</code>.
     * 
     * @param clasz
     *            Class to create an instance for.
     * @param argTypes
     *            Argument types of the constructor to use.
     * @param initArgs
     *            Argument values for the constructor.
     * 
     * @return New instance.
     */
    @SuppressWarnings("unchecked")
    public static Object createInstance(final SgClass clasz, final Class[] argTypes,
            final Object[] initArgs) {

        final Class newClass = createClass(clasz);
        return createInstance(newClass, argTypes, initArgs);

    }

    /**
     * Creates an instance from a model class with it's no argument constructor
     * and maps all exceptions into <code>RuntimeException</code>.
     * 
     * @param clasz
     *            Class to create an instance for.
     * 
     * @return New instance.
     */
    @SuppressWarnings("unchecked")
    public static Object createInstance(final SgClass clasz) {

        final Class newClass = createClass(clasz);
        return createInstance(newClass, new Class[] {}, new Object[] {});

    }

}
