/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.objects4j;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Label information for a concrete object.
 */
@Immutable
public abstract class LabelInfo {

    private final String text;

    private final String shortText;

    /**
     * Constructor with text and short text.
     * 
     * @param text
     *            Text.
     * @param shortText
     *            Abbreviation of the text.
     */
    public LabelInfo(final String text, final String shortText) {
        super();
        this.text = text;
        this.shortText = shortText;
    }

    /**
     * Returns the text of the label.
     * 
     * @return Long text.
     */
    public final String getText() {
        return text;
    }

    /**
     * Returns the abbreviation of the text.
     * 
     * @return Short text.
     */
    public final String getShortText() {
        return shortText;
    }

    /**
     * Returns the text for the label. If no entry is found in the resource
     * bundle for {@link Label#key()} then {@link Label#value()} will be
     * returned instead. If {@link Label#value()} is also empty then
     * <code>null</code> is returned. If {@link Label#key()} is empty
     * <code>defaultKey</code> will be used as key in the properties file.
     * 
     * @param bundle
     *            Resource bundle to use.
     * @param label
     *            Label for the field.
     * @param defaultKey
     *            Default key if {@link Label#key()} is empty.
     * 
     * @return Text or <code>null</code>.
     */
    @Requires("(bundle != null) && (label != null) && (field!=null)")
    public static String getText(final ResourceBundle bundle, final Label label,
            final String defaultKey) {

        Contract.requireArgNotNull("bundle", bundle);
        Contract.requireArgNotNull("label", label);
        Contract.requireArgNotNull("defaultKey", defaultKey);

        final String key;
        if (label.key().equals("")) {
            key = defaultKey;
        } else {
            key = label.key();
        }

        try {
            return bundle.getString(key);
        } catch (final MissingResourceException ex) {
            return toNullableString(label.value());
        }

    }

    /**
     * Returns the short text for the label. If no entry is found in the
     * resource bundle for {@link Label#shortKey()} then
     * {@link Label#shortText()} will be returned instead. If
     * {@link Label#shortText()} is also empty then <code>null</code> is
     * returned. If {@link Label#shortKey()} is empty <code>defaultKey</code>
     * will be used as key in the properties file.
     * 
     * @param bundle
     *            Resource bundle to use.
     * @param label
     *            Label for the field.
     * @param defaultKey
     *            Default key if {@link Label#shortKey()} is empty.
     * 
     * @return Text or <code>null</code>.
     */
    @Requires("(bundle != null) && (label != null) && (field!=null)")
    public static String getShortText(final ResourceBundle bundle, final Label label,
            final String defaultKey) {

        Contract.requireArgNotNull("bundle", bundle);
        Contract.requireArgNotNull("label", label);
        Contract.requireArgNotNull("field", defaultKey);

        final String key;
        if (label.shortKey().equals("")) {
            key = defaultKey;
        } else {
            key = label.shortKey();
        }

        try {
            return bundle.getString(key);
        } catch (final MissingResourceException ex) {
            return toNullableString(label.shortText());
        }

    }

    /**
     * Returns the resource bundle for a given label. If
     * <code>label.bundle()</code> is empty the <code>clasz</code> is used to
     * create a path and filename information.
     * 
     * @param label
     *            Label with bundle name.
     * @param locale
     *            Locale to use.
     * @param clasz
     *            Class to use if the <code>label.bundle()</code> is empty.
     *            Example: <code>a.b.c.MyClass</code> is used as
     *            <code>a/b/c/MyClass.properties</code> or
     *            <code>a/b/c/MyClass_en.properties</code> (with
     *            {@link Locale#ENGLISH}).
     * 
     * @return Resource bundle.
     */
    @Requires("(label != null) && (locale != null) && (clasz != null)")
    @Ensures("\result != null")
    public static ResourceBundle getResourceBundle(final Label label, final Locale locale,
            final Class<?> clasz) {

        if (label.bundle().equals("")) {
            final String path = clasz.getPackage().getName().replace('.', '/');
            final String baseName = path + "/" + clasz.getSimpleName();
            return ResourceBundle.getBundle(baseName, locale);
        }
        return ResourceBundle.getBundle(label.bundle(), locale);

    }

    protected static String toNullableString(final String value) {
        if (value.equals("")) {
            return null;
        }
        return value;
    }

}
