/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.objects4j;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Label information for a class.
 */
@Immutable
public class LabelClassInfo extends LabelInfo {

    private final Class<?> clasz;

    /**
     * Constructor with class, text and abbreviation.
     * 
     * @param clasz
     *            Class the label is for.
     * @param text
     *            Text.
     * @param shortText
     *            Abbreviation of the text.
     */
    @Requires("clasz != null")
    public LabelClassInfo(final Class<?> clasz, final String text, final String shortText) {
        super(text, shortText);
        Contract.requireArgNotNull("clasz", clasz);
        this.clasz = clasz;
    }

    /**
     * Returns the class the label is for.
     * 
     * @return Class.
     */
    public final Class<?> getClasz() {
        return clasz;
    }

    @Override
    public final int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + clasz.getName().hashCode();
        return result;
    }

    @Override
    public final boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final LabelClassInfo other = (LabelClassInfo) obj;
        return clasz.getName().equals(other.clasz.getName());
    }

    /**
     * Returns the label information for a given class.
     * 
     * @param clasz
     *            Class.
     * @param locale
     *            Locale to use.
     * 
     * @return Label information or <code>null</code>.
     */
    @Requires("(clasz != null) && (locale != null)")
    public static LabelClassInfo create(final Class<?> clasz, final Locale locale) {

        Contract.requireArgNotNull("clasz", clasz);
        Contract.requireArgNotNull("locale", locale);

        final Label label = clasz.getAnnotation(Label.class);
        if (label == null) {
            return null;
        }

        try {
            final ResourceBundle bundle = getResourceBundle(label, locale, clasz);
            final String text = getText(bundle, label, clasz.getSimpleName());
            final String shortText = getShortText(bundle, label, clasz.getSimpleName() + ".short");
            return new LabelClassInfo(clasz, text, shortText);
        } catch (final MissingResourceException ex) {
            if (label.value().equals("")) {
                return new LabelClassInfo(clasz, null, null);
            }
            return new LabelClassInfo(clasz, label.value(), null);
        }

    }

}
