/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.objects4j;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Label information for a field of a class.
 */
@Immutable
public final class LabelFieldInfo extends LabelInfo {

    private final String field;

    /**
     * Constructor with field, text and abbreviation.
     * 
     * @param field
     *            Name of the field.
     * @param text
     *            Text.
     * @param shortText
     *            Abbreviation of the text.
     */
    @Requires("field != null")
    public LabelFieldInfo(final String field, final String text, final String shortText) {
        super(text, shortText);
        Contract.requireArgNotNull("field", field);
        this.field = field;
    }

    /**
     * Returns the name of the field.
     * 
     * @return Name.
     */
    @Ensures("\result != null")
    public final String getField() {
        return field;
    }

    /**
     * Returns the text of the label or the name of the field if the text is
     * <code>null</code>.
     * 
     * @return Long text or field name.
     */
    @Ensures("\result != null")
    public final String getTextOrField() {
        final String text = getText();
        if (text == null) {
            return field;
        }
        return text;
    }

    @Override
    public final int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + field.hashCode();
        return result;
    }

    @Override
    public final boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final LabelFieldInfo other = (LabelFieldInfo) obj;
        return field.equals(other.field);
    }

    /**
     * Returns label informations for all field of a class that are annotated
     * with {@link Label}. All other fields ar ignored.
     * 
     * @param clasz
     *            Class that contains the fields.
     * @param locale
     *            Locale to use.
     * 
     * @return List of label informations.
     */
    @Requires("(clasz != null) && (locale != null)")
    @Ensures("\result != null")
    public static List<LabelFieldInfo> create(final Class<?> clasz, final Locale locale) {

        final List<LabelFieldInfo> labelInfos = new ArrayList<LabelFieldInfo>();

        final Field[] fields = clasz.getDeclaredFields();
        for (final Field field : fields) {
            final Label label = field.getAnnotation(Label.class);
            if (label != null) {
                try {
                    final ResourceBundle bundle = getResourceBundle(label, locale, field
                            .getDeclaringClass());
                    final String text = getText(bundle, label, field.getName());
                    final String shortText = getShortText(bundle, label, field.getName() + ".short");
                    labelInfos.add(new LabelFieldInfo(field.getName(), text, shortText));
                } catch (final MissingResourceException ex) {
                    final String text = toNullableString(label.value());
                    final String shortText = toNullableString(label.shortText());
                    labelInfos.add(new LabelFieldInfo(field.getName(), text, shortText));
                }
            }
        }

        Class<?> parent = clasz;
        while ((parent = parent.getSuperclass()) != Object.class) {
            labelInfos.addAll(create(parent, locale));
        }

        return labelInfos;

    }

    /**
     * Returns the label information for a given field of a class.
     * 
     * @param clasz
     *            Class that contains the field.
     * @param fieldName
     *            Name of the field within the class.
     * @param locale
     *            Locale to use.
     * 
     * @return Label information.
     * 
     * @throws NoSuchFieldException
     *             There is no field of that name inside the class.
     */
    @Requires("(clasz != null) && (fieldName != null) && (locale != null)")
    @Ensures("\result != null")
    public static LabelFieldInfo create(final Class<?> clasz, final String fieldName,
            final Locale locale) throws NoSuchFieldException {

        final Field field = clasz.getDeclaredField(fieldName);
        field.setAccessible(true);
        return create(field, locale);

    }

    /**
     * Returns the label information for a given field of a class.
     * 
     * @param field
     *            Field within the class.
     * @param locale
     *            Locale to use.
     * 
     * @return Label information.
     */
    @Requires("(field != null) && (locale != null)")
    @Ensures("\result != null")
    public static LabelFieldInfo create(final Field field, final Locale locale) {

        Contract.requireArgNotNull("field", field);
        Contract.requireArgNotNull("locale", locale);

        final Label label = field.getAnnotation(Label.class);
        if (label == null) {
            return new LabelFieldInfo(field.getName(), null, null);
        }

        try {
            final ResourceBundle bundle = getResourceBundle(label, locale, field
                    .getDeclaringClass());
            final String text = getText(bundle, label, field.getName());
            final String shortText = getShortText(bundle, label, field.getName());
            return new LabelFieldInfo(field.getName(), text, shortText);
        } catch (final MissingResourceException ex) {
            return new LabelFieldInfo(field.getName(), null, null);
        }
    }

    @Override
    public String toString() {
        return field;
    }

}
