/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.objects4j;

import java.io.Serializable;

import javax.validation.constraints.NotNull;

import org.fuin.objects4j.validation.UserIdStr;

/**
 * UserId with the following rules.
 * <ul>
 * <li>3-20 characters in length</li>
 * <li>Lower case letters (a-z)</li>
 * <li>Numbers (0-9)</li>
 * <li>Hyphens (-)</li>
 * <li>Underscores (_)</li>
 * <li>Starts not with an underscore, hyphen or number</li>
 * </ul>
 */
@Immutable
public final class UserId implements Comparable<UserId>, Serializable {

    private static final long serialVersionUID = 9055520843135472634L;

    @NotNull
    @UserIdStr
    private final String userId;

    /**
     * Constructor with userId.
     * 
     * @param userId
     *            UserId.
     */
    @Requires("(userId!=null) && ValidUserIdValidator.isValid(userId)")
    public UserId(final String userId) {
        super();
        this.userId = userId.trim().toLowerCase();
        Contract.requireValid(this);
    }

    // CHECKSTYLE:OFF Generated hashCode + equals
    @Override
    public final int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((userId == null) ? 0 : userId.hashCode());
        return result;
    }

    @Override
    public final boolean equals(final Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        UserId other = (UserId) obj;
        if (userId == null) {
            if (other.userId != null)
                return false;
        } else if (!userId.equals(other.userId))
            return false;
        return true;
    }

    // CHECKSTYLE:ON

    /**
     * Returns the length of the user id.
     * 
     * @return Number of characters.
     */
    public final int length() {
        return userId.length();
    }

    @Override
    public final int compareTo(final UserId obj) {
        final UserId other = (UserId) obj;
        return this.userId.compareTo(other.userId);
    }

    @Override
    public String toString() {
        return userId;
    }

}
