/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.objects4j;

import java.io.Serializable;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

/**
 * SHA-512 hashed password.
 */
@Immutable
public final class PasswordSha512 implements Serializable {

    private static final long serialVersionUID = -6285061339408965704L;

    private static final char[] DIGITS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a',
            'b', 'c', 'd', 'e', 'f' };

    private final byte[] hash;

    // Computed on demand
    private transient String hashStr;

    /**
     * Constructor with password.
     * 
     * @param password
     *            Clear text password.
     */
    @Requires("password!=null")
    public PasswordSha512(final Password password) {
        super();
        try {
            final MessageDigest md = MessageDigest.getInstance("SHA-512");
            md.update(password.toString().getBytes());
            this.hash = md.digest();
        } catch (final NoSuchAlgorithmException ex) {
            throw new RuntimeException("", ex);
        }
    }

    @Override
    public final String toString() {
        if (hashStr == null) {
            final int l = hash.length;
            final char[] out = new char[l << 1];
            int j = 0;
            for (int i = 0; i < l; i++) {
                out[j++] = DIGITS[(0xF0 & hash[i]) >>> 4];
                out[j++] = DIGITS[0x0F & hash[i]];
            }
            hashStr = String.copyValueOf(out);
        }
        return hashStr;
    }

    // CHECKSTYLE:OFF Generated hashCode + equals

    @Override
    public final int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + Arrays.hashCode(hash);
        return result;
    }

    @Override
    public final boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final PasswordSha512 other = (PasswordSha512) obj;
        if (!Arrays.equals(hash, other.hash)) {
            return false;
        }
        return true;
    }

    // CHECKSTYLE:ON

}
