/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URL;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.swing.JOptionPane;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.launcher.CommandLauncher;
import org.apache.commons.exec.launcher.CommandLauncherFactory;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.fuin.utils4j.Cancelable;
import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.common.Utils4Swing;
import org.fuin.utils4swing.dialogs.CanceledException;
import org.fuin.utils4swing.dialogs.DirectorySelector;
import org.fuin.utils4swing.progress.FileCopyProgressListener;
import org.fuin.utils4swing.progress.FileCopyProgressMonitor;
import org.fuin.utils4swing.threadsafe.ThreadSafeJOptionPane;

/**
 * Main application.
 */
public final class Kickstart4J {

	private static final Logger LOG = Logger.getLogger(Kickstart4J.class);

	private static final String PROGRAM_TERMINATED_WITH_ERROR = "Program terminated with error!";

	private static final String INCOMPLETE_FILE = ".incomplete";

	private static final String PROGRAM_DIRECTORY_KEY = "program-directory";

	/** Configuration used for the application. */
	private final Config config;

	/** Listens to life cycle events. */
	private Kickstart4JListener listener;
	
	/**
	 * Constructor with configuration.
	 * 
	 * @param config
	 *            Configuration to use.
	 */
	public Kickstart4J(final Config config) {
		super();
		if (config == null) {
			throw new IllegalArgumentException("The argument 'config' cannot be null!");
		}
		this.config = config;
		this.listener = new DefaultListener();
	}

	/**
	 * Returns the life cycle listener.
	 * 
	 * @return Listener - Always non-<code>null</code>.
	 */
	public final Kickstart4JListener getListener() {
		return listener;
	}

	/**
	 * Sets the life cycle listener.
	 * 
	 * @param listener Listener - Will be set to a default listener if <code>null</code>.
	 */
	public final void setListener(final Kickstart4JListener listener) {
		if (listener == null) {
			this.listener = new DefaultListener();
		} else {
			this.listener = listener;
		}
	}

	/**
	 * Initialize file logging with configuration values.
	 */
	private void initLogging() {

		try {
			final File logFile = new File(config.getLogFilename()).getCanonicalFile();

			boolean ok = true;
			if (!logFile.getParentFile().exists()) {
				ok = logFile.getParentFile().mkdirs();
			}

			if (ok) {
				final Properties props = new Properties();
				props.put("log4j.rootLogger", "INFO, FILE");
				props.put("log4j.appender.FILE", "org.apache.log4j.RollingFileAppender");
				props.put("log4j.appender.FILE.File", logFile.toString());
				props.put("log4j.appender.FILE.MaxFileSize", "1MB");
				props.put("log4j.appender.FILE.MaxBackupIndex", "1");
				props.put("log4j.appender.FILE.layout", "org.apache.log4j.PatternLayout");
				props.put("log4j.appender.FILE.layout.ConversionPattern",
						"%d [%t] %-5p %c - %m%n");
				PropertyConfigurator.configure(props);
			} else {
				LOG.error("Cannot create log directory: " + logFile.getParentFile());
			}
		} catch (final IOException ex) {
			LOG.error("Cannot create log!", ex);
		}

	}

	/**
	 * Executes the installer/updater.
	 * 
	 * @throws CanceledException
	 *             The user canceled the installation.
	 * @throws InvalidConfigException
	 *             The configuration is invalid.
	 */
	public final void execute() throws CanceledException, InvalidConfigException {

		Locale.setDefault(config.getLocale());
		final File destDir = getDestDir();
		config.getCmdLineOptions().put("destDir", destDir.toString());

		// Check config AFTER destination dir is set and init logging
		config.check();
		initLogging();
		listener.initComplete();

		// Start the update
		final UpdateSet updateSet = new UpdateSet(config.getSrcFiles(), config.getMkDirs(),
				destDir, config.isLazyLoading());
		if (updateSet.isUpdateNecessary()) {
			if (LOG.isInfoEnabled()) {
				LOG.info("An update is available: New=" + updateSet.getNewFiles().size()
						+ ", Changed=" + updateSet.getChangedFiles().size() + ", Deleted="
						+ updateSet.getDeletedFiles().size() + ", SilentInstall="
						+ config.isSilentInstall() + ", SilentUpdate="
						+ config.isSilentUpdate() + ", FirstInstallation="
						+ config.isFirstInstallation());
			}
			if (config.isSilentUpdate() || config.isFirstInstallation()
					|| isAnswerYes(config.getMessages().getUpdateAvailable())) {
				execute(updateSet);
				final File installationIncompleteFile = new File(destDir, INCOMPLETE_FILE);
				if (installationIncompleteFile.exists()) {
					installationIncompleteFile.delete();
				}
			}
		} else {
			LOG.info("Files are up to date");
		}
		config.getCmdLineOptions().put("classpath", updateSet.createClasspath());

		// Write the config to the target directory
		saveConfigToTargetDir(destDir);

		// Run the target application
		final CommandLine commandLine = new CommandLine(new File(config.getDestDir(), config
				.getJavaExe()));
		commandLine.addArguments(config.getJavaArgs(), false);
		logStart(destDir, commandLine.toString());
		startExternal(destDir, commandLine);

	}

	private void saveConfigToTargetDir(final File destDir) {
		final File appXmlFile = new File(destDir, "application.xml");
		try {
			final String localConfigFileUrl = appXmlFile.toURI().toURL().toString();
			config.setConfigFileUrl(localConfigFileUrl);
			config.getCmdLineOptions().put("configFileUrl", localConfigFileUrl);
			config.writeToStaticXML(appXmlFile, true);
		} catch (final IOException ex) {
			throw new RuntimeException("Error writing " + appXmlFile + "!", ex);
		}
	}

	private File getIdFile(final Config config) {
		return new File(Utils4J.getUserHomeDir(), config.getIdFilename());
	}

	private void execute(final UpdateSet updateSet) throws CanceledException {
		executeMkdirs(updateSet);
		executeCopy(updateSet);
		executeDecompress(updateSet);
	}

	private void executeMkdirs(final UpdateSet updateSet) {

		for (int i = 0; i < updateSet.getMkDirs().size(); i++) {
			final MkDir mkDir = (MkDir) updateSet.getMkDirs().get(i);
			final File destDir = mkDir.getDestDir(updateSet.getDestDir());
			if (destDir.exists()) {
				LOG.info("MKDIR: " + destDir + " (Already exists)");
			} else {
				final boolean ok = destDir.mkdirs();
				if (LOG.isInfoEnabled()) {
					if (ok) {
						LOG.info("MKDIR: " + destDir);
					} else {
						LOG.info("MKDIR FAILED: " + destDir);
					}
				}
			}
		}

	}

	private void executeCopy(final UpdateSet updateSet) throws CanceledException {

		final int max = updateSet.getNewFiles().size() + updateSet.getChangedFiles().size();
		if (max > 0) {

			final Cancelable cancelable = new Cancelable() {

				private volatile boolean canceled = false;

				public void cancel() {
					canceled = true;
				}

				public boolean isCanceled() {
					return canceled;
				}
			};

			final FileCopyProgressMonitor monitor = new FileCopyProgressMonitor(cancelable,
					config.getTitle(), config.getMessages().getProgressMonitorTransferText(),
					config.getMessages().getProgressMonitorSrcLabelText(), config
							.getMessages().getProgressMonitorDestLabelText(), max);

			int count = 0;

			monitor.open();
			try {

				// New files
				if (cancelable.isCanceled()) {
					throw new CanceledException();
				}
				count = copyFiles(updateSet, cancelable, monitor, updateSet.getNewFiles(),
						count, "NEW");

				// Changed files
				if (cancelable.isCanceled()) {
					throw new CanceledException();
				}
				count = copyFiles(updateSet, cancelable, monitor, updateSet.getChangedFiles(),
						count, "CHANGED");

				// No longer existent (deleted) files
				if (cancelable.isCanceled()) {
					throw new CanceledException();
				}
				for (int i = 0; i < updateSet.getDeletedFiles().size(); i++) {
					if (cancelable.isCanceled()) {
						break;
					}
					count = count + 1;
					final String file = (String) updateSet.getDeletedFiles().get(i);
					final File destFile = new File(updateSet.getDestDir(), file);
					final boolean ok = destFile.delete();
					if (LOG.isInfoEnabled()) {
						if (ok) {
							LOG.info("DELETED: " + destFile);
						} else {
							LOG.info("DELETE FAILED: " + destFile);
						}
					}
					monitor.updateFile("", destFile.toString(), count, 0);
				}

			} finally {
				monitor.close();
			}

		}
	}

	private void executeDecompress(final UpdateSet updateSet) throws CanceledException {

		final List compressedFiles = updateSet.getDecompressFiles();
		final int max = compressedFiles.size();
		if (max > 0) {

			final Cancelable cancelable = new Cancelable() {

				private volatile boolean canceled = false;

				public void cancel() {
					canceled = true;
				}

				public boolean isCanceled() {
					return canceled;
				}
			};

			final FileCopyProgressMonitor monitor = new FileCopyProgressMonitor(cancelable,
					config.getTitle(),
					config.getMessages().getProgressMonitorDecompressText(), config
							.getMessages().getProgressMonitorSrcLabelText(), config
							.getMessages().getProgressMonitorDestLabelText(), max);

			monitor.open();
			try {

				for (int i = 0; i < max; i++) {
					final SrcFile file = (SrcFile) compressedFiles.get(i);
					final File compressedFile = file.getDestFile(updateSet.getDestDir());
					if (LOG.isInfoEnabled()) {
						LOG.info("Decompressing: " + compressedFile);
					}
					Utils.unzip(monitor, compressedFile, (i + 1), updateSet.getDestDir(),
							cancelable);
				}

			} finally {
				monitor.close();
			}

		}
	}

	private int copyFiles(final UpdateSet updateSet, final Cancelable cancelable,
			final FileCopyProgressListener listener, final List files, final int total,
			final String type) {

		int count = total;

		for (int i = 0; i < files.size(); i++) {
			if (cancelable.isCanceled()) {
				break;
			}
			count = count + 1;
			final SrcFile file = (SrcFile) files.get(i);
			final URL srcFileUrl = file.getSrcFileURL();
			final File destFile = file.getDestFile(updateSet.getDestDir());
			try {
				Utils
						.copyURLToFile(listener, srcFileUrl, destFile, count, file
								.getSizeAsInt());
			} catch (final FileNotFoundException ex) {
				throw new RuntimeException("Source file not found!", ex);
			}
			final String hash = Utils4J.createHashMD5(destFile);
			if (!hash.equals(file.getMd5Hash())) {
				LOG.error("Hash local file (" + hash
						+ ") is different from configuration hash (" + file.getMd5Hash()
						+ ")! [" + srcFileUrl + "]");
			}
			if (LOG.isInfoEnabled()) {
				LOG.info(type + ": " + srcFileUrl + " => " + destFile);
			}
		}

		return count;

	}

	private void logStart(final File dir, final String commandLine) {
		try {
			final File file = new File(dir, "start.log");
			final FileWriter writer = new FileWriter(file);
			try {
				writer.write(commandLine);
			} finally {
				writer.close();
			}
		} catch (final IOException ex) {
			throw new RuntimeException(ex);
		}
	}

	private void startExternal(final File destDir, final CommandLine commandLine) {
		final CommandLauncher launcher = CommandLauncherFactory.createVMLauncher();
		try {
			launcher.exec(commandLine, null, destDir);
			// TODO Get errors from new process if it does not start
			// TODO Display something like "Starting application..." for X
			// seconds
			if (config.isExitAfterExecute()) {
				System.exit(0);
			}
		} catch (final Exception ex) {
			throw new RuntimeException(ex);
		}
	}

	private File getDestDir() throws CanceledException {
		final File installationIncompleteFile;
		final File dir;
		final File idFile = getIdFile(config);
		if (idFile.exists()) {
			// Update
			final Properties props = Utils4J.loadProperties(idFile);
			final String dirStr = props.getProperty(PROGRAM_DIRECTORY_KEY);
			if (dirStr == null) {
				throw new IllegalStateException("The property '" + PROGRAM_DIRECTORY_KEY
						+ "' was not found inside '" + idFile + "'!");
			}
			dir = new File(dirStr);
			Utils4J.checkValidDir(dir);
			installationIncompleteFile = new File(dir, INCOMPLETE_FILE);
		} else {
			// First installation or file removed
			final String dirStr;
			final Properties props = new Properties();
			if (config.isSilentInstall()) {
				dirStr = config.getDestDir().toString();
			} else {
				// Ask User for destination directory
				dirStr = DirectorySelector.selectDirectory(
						config.getMessages().getSelectDestinationDirectory(),
						config.getDestDir().toString()).getDirectory();
			}
			props.setProperty(PROGRAM_DIRECTORY_KEY, dirStr);
			Utils4J.saveProperties(idFile, props,
					"# --- DO NOT EDIT OR DELETE --- Generated by Kickstart4J ---");
			dir = new File(dirStr);
			installationIncompleteFile = new File(dir, INCOMPLETE_FILE);
			if (!dir.exists()) {
				try {
					dir.mkdirs();
					installationIncompleteFile.createNewFile();
				} catch (final IOException ex) {
					throw new RuntimeException("Cannot create file '"
							+ installationIncompleteFile + "'!");
				}
			}
		}
		config.setFirstInstallation(installationIncompleteFile.exists());
		return dir;
	}

	private static boolean isAnswerYes(final String message) {
		final int result = ThreadSafeJOptionPane.showConfirmDialog(null, message, "TITLE",
				JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
		return result == JOptionPane.YES_OPTION;
	}

	private static void showError(final String message) {
		ThreadSafeJOptionPane.showMessageDialog(null, message, "Error",
				JOptionPane.ERROR_MESSAGE);
	}

	private static void showMessage(final String message) {
		ThreadSafeJOptionPane.showMessageDialog(null, message, "Hint",
				JOptionPane.INFORMATION_MESSAGE);
	}

	private static void displayCmdLineException(final CmdLineParser parser,
			final CmdLineException ex) {
		final StringBuffer sb = new StringBuffer();
		sb.append(ex.getMessage());
		sb.append("\n");
		sb.append("\n");
		sb.append("java Kickstart4J [options]");
		sb.append("\n");
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		parser.printUsage(out);
		sb.append(out.toString());
		sb.append("\n");
		System.out.println(sb);
		showError(sb.toString());
	}

	private static void displayException(final Exception ex) {
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		ex.printStackTrace(new PrintStream(out));
		final String msg = out.toString();
		System.err.println(msg);
		showError(msg);
	}
	
	/**
	 * Empty implementation.  
	 */
	private static final class DefaultListener implements Kickstart4JListener {

		/**
		 * {@inheritDoc}
		 */
		public final void initComplete() {
			// Do nothing
		}
		
	}

	/**
	 * Main method used to start the installer/updater. If you want to start it
	 * from another Java application you can simply use
	 * <code>new Kickstart4J(kickstart4JConfig).execute()</code> instead of
	 * calling this method.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(final String[] args) {

		final Logger log = Logger.getLogger(Kickstart4J.class);

		final Config config = new Config();
		try {

			// Parse command line
			final CmdLineParser cmdLineParser = new CmdLineParser(Locale.getDefault());
			try {
				cmdLineParser.parse(args);
				if (log.isDebugEnabled()) {
					log.info("Command line arguments: " + cmdLineParser);
				}

				// Set user defined options from command line
				cmdLineParser.copyToConfig(config);

				// Load the configuration and start update
				try {
					ConfigParser.parse(config, config.getConfigFileURL());
					if (log.isInfoEnabled()) {
						log.info("Configuration: " + config);
					}

					Utils4Swing.initLookAndFeel(config.getLookAndFeelClassName());
					(new Kickstart4J(config)).execute();
					log.info("Operation finished successfully!");
					System.exit(0);

				} catch (final CanceledException ex) {
					log.info("Operation canceled by user!");
					showMessage(config.getMessages().getOperationCanceled());
					System.exit(0);
				} catch (final InvalidConfigException ex) {
					log.error(PROGRAM_TERMINATED_WITH_ERROR, ex);
					displayException(ex);
					System.exit(1);
				}

			} catch (final CmdLineException ex) {
				log.error(PROGRAM_TERMINATED_WITH_ERROR, ex);
				displayCmdLineException(cmdLineParser, ex);
				System.exit(1);
			}

		} catch (final RuntimeException ex) {
			log.error(PROGRAM_TERMINATED_WITH_ERROR, ex);
			displayException(ex);
			System.exit(1);
		}

	}

}
