/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.fuin.utils4j.Utils4J;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Parses a XML configuration.
 */
public final class Kickstart4JConfigParser {

    private final ConfigHandler handler;

    /**
     * Default constructor.
     */
    public Kickstart4JConfigParser() {
        super();
        handler = new ConfigHandler();
    }

    /**
     * Starts the parsing process.
     * 
     * @param urlStr
     *            URL of the XML configuration file.
     * 
     * @throws InvalidConfigException
     *             Error parsing the configuration.
     */
    public void parse(final String urlStr) throws InvalidConfigException {
        try {
            final URL url = new URL(urlStr);
            try {
                final SAXParserFactory factory = SAXParserFactory.newInstance();
                factory.setValidating(false);
                factory.newSAXParser().parse(url.toExternalForm(), handler);
            } catch (final SAXException e) {
                throw new InvalidConfigException(url, e);
            } catch (final ParserConfigurationException e) {
                throw new InvalidConfigException(url, e);
            } catch (final IOException e) {
                throw new InvalidConfigException(url, e);
            }
        } catch (final MalformedURLException ex) {
            throw new InvalidConfigException("Invalid URL '" + urlStr + "'!");
        }
    }

    /**
     * Copies the values from the handler into the config.
     * 
     * @param config
     *            Configuration to use.
     * 
     * @throws InvalidConfigException
     *             Error copying the values.
     */
    public void copyToConfig(final Kickstart4JConfig config) throws InvalidConfigException {

        final List elements = handler.getElements();
        for (int i = 0; i < elements.size(); i++) {
            final ConfigElement element = (ConfigElement) elements.get(i);
            if (element.getName().equals("file")) {
                final Map atts = element.getAttributes();
                final String path = (String) atts.get("path");
                checkNotNull(element, "path", path);
                final String file = (String) atts.get("file");
                checkNotNull(element, "file", file);
                final String hash = (String) atts.get("hash");
                checkNotNull(element, "hash", hash);
                final String size = (String) atts.get("size");
                checkNotNull(element, "size", size);
                final boolean unzip = getBoolean(atts, "unzip", false);
                final boolean loadAlways = getBoolean(atts, "loadAlways", false);
                final boolean addToClasspath = getBoolean(atts, "addToClasspath", false);
                config.getSrcFiles().add(
                        new SrcFile(path, file, hash, Long.valueOf(size).longValue(), unzip,
                                loadAlways, addToClasspath));
            } else {
                config.put(element.getName(), element.getText());
            }
        }

    }

    private static void checkNotNull(final ConfigElement element, final String attrName,
            final String attrValue) throws InvalidConfigException {
        if (attrValue == null) {
            throw new InvalidConfigException("Element '" + element.getName()
                    + "' missing required attribute '" + attrName + "'!");
        }
    }

    private static boolean getBoolean(final Map atts, final String key,
            final boolean defaultValue) {
        final String value = (String) atts.get(key);
        if (value == null) {
            return defaultValue;
        }
        final String str = value.trim();
        if (str.length() == 0) {
            return defaultValue;
        }
        return Boolean.valueOf(str).booleanValue();
    }

    /**
     * Element in the configuration.
     */
    private static final class ConfigElement {

        private final String name;

        private final Map attributes;

        private String text = null;

        public ConfigElement(final String name, final Attributes atts) {
            super();

            this.name = name;
            this.attributes = new HashMap();

            final int length = atts.getLength();
            for (int i = 0; i < length; i++) {
                attributes.put(atts.getQName(i), atts.getValue(i));
            }
        }

        public final Map getAttributes() {
            return attributes;
        }

        public final String getName() {
            return name;
        }

        public final String getText() {
            return text;
        }

        public final void setText(final String text) {
            if (text == null) {
                this.text = null;
            } else {
                this.text = text.trim();
                if (this.text.length() == 0) {
                    this.text = null;
                }
            }
        }

        public final boolean getTextAsBoolean() {
            if ((text == null) || (text.length() == 0)) {
                return false;
            }
            return Boolean.valueOf(text).booleanValue();
        }

        public String toString() {
            final StringBuffer sb = new StringBuffer();
            sb.append(name);
            sb.append("=");
            sb.append(text);
            sb.append(" {");
            int count = 0;
            final Iterator it = attributes.keySet().iterator();
            while (it.hasNext()) {
                final String key = (String) it.next();
                final String value = (String) attributes.get(key);
                if (count > 0) {
                    sb.append(", ");
                }
                sb.append(key);
                sb.append("=");
                sb.append(value);
                count++;
            }
            sb.append("}");
            return sb.toString();
        }

    }

    /**
     * Handler for parsing the XML.
     */
    private static class ConfigHandler extends DefaultHandler {

        private final List elements = new ArrayList();

        private ConfigElement element = null;

        private int level = 0;

        public void startElement(final String uri, final String localName, final String qName,
                final Attributes atts) throws SAXException {

            if (level == 1) {
                element = new ConfigElement(qName, atts);
                elements.add(element);
            }
            level++;

        }

        public void characters(final char[] ch, final int start, final int length)
                throws SAXException {
            if ((level == 2) && (element != null)) {
                element.setText(String.copyValueOf(ch, start, length));
            }
        }

        public void endElement(final String uri, final String localName, final String qName)
                throws SAXException {
            if (level == 1) {
                element = null;
            }
            level--;
        }

        public List getElements() {
            return elements;
        }

    }

    /**
     * Loads a config from a XML configuration file.
     * 
     * @param config
     *            Configuration to populate.
     * @param configFileURL
     *            URL of the configuration file.
     * 
     * @return Configuration filled with values from the config file.
     * 
     * @throws InvalidConfigException
     *             Error reading the file.
     */
    public static Kickstart4JConfig parse(final Kickstart4JConfig config,
            final String configFileURL) throws InvalidConfigException {

        Utils4J.checkNotNull("config", config);
        Utils4J.checkNotNull("configFileURL", configFileURL);

        final Kickstart4JConfigParser parser = new Kickstart4JConfigParser();
        parser.parse(configFileURL);
        parser.copyToConfig(config);
        return config;

    }

    /**
     * Creates a config from a XML configuration file.
     * 
     * @param configFileURL
     *            URL of the configuration file.
     * 
     * @return Configuration with values from the config file.
     * 
     * @throws InvalidConfigException
     *             Error reading the file.
     */
    public static Kickstart4JConfig create(final String configFileURL)
            throws InvalidConfigException {

        Utils4J.checkNotNull("configFileURL", configFileURL);

        return parse(new Kickstart4JConfig(), configFileURL);

    }

    /**
     * Creates a config from a XML configuration file.
     * 
     * @param configFile
     *            Configuration file.
     * 
     * @return Configuration with values from the config file.
     * 
     * @throws InvalidConfigException
     *             Error reading the file.
     */
    public static Kickstart4JConfig create(final File configFile)
            throws InvalidConfigException {
        try {
            return create(configFile.toURI().toURL().toExternalForm());
        } catch (final MalformedURLException ex) {
            throw new RuntimeException("Error creating file URL!", ex);
        }
    }

}
