/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;

import javax.swing.UIManager;

import org.apache.commons.io.IOUtils;
import org.fuin.utils4j.Utils4J;

/**
 * Configuration for the application.
 */
public final class Kickstart4JConfig {

    /** Fully qualified LnF class name. */
    private String lookAndFeelClassName = UIManager.getSystemLookAndFeelClassName();

    /** Unique id filename for the application. */
    private String idFilename;

    /** Determines if the program terminates with <code>System.exit(..)</code>. */
    private boolean exitAfterExecute = true;

    /** URL of the directory where the source files are located. */
    private URL srcUrl = null;

    /** Local directory where the files should be copied to. */
    private File destDir = null;

    /** Determines if the user should be prompted for the destination directory. */
    private boolean silentInstall = false;

    /** Determines if the user should be prompted if an update is available. */
    private boolean silentUpdate = false;

    /** Determines if this is the first installation. */
    private boolean firstInstallation = true;

    /** Locale to use. */
    private Locale locale = Locale.getDefault();

    /** File containing the localized installation messages. */
    private String msgFile = null;

    /** Message properties needed for the installer. */
    private Kickstart4JMessages messages = null;

    /** Load only the "jar" entries on startup? */
    private boolean lazyLoading = false;

    /** Additional user defined options. */
    private Map userDefinedOptions = new HashMap();

    /** Java executable. */
    private String javaExe = null;

    /** Commandline including (without java executable itself). */
    private String javaArgs = null;

    /** Title of the application. */
    private String title = null;

    /** Vendor of the application. */
    private String vendor = null;

    /** Short description of the application. */
    private String description = null;

    /** List of known files. */
    private List srcFiles = new ArrayList();

    /** URL of the config file (only used when lazyLoading=true). */
    private String configFileUrlStr = null;

    /** Target application version. */
    private String version = null;

    /**
     * Returns the name of the LookAndFeel class.
     * 
     * @return Fully qualified LnF class name - Always non-null.
     */
    public final String getLookAndFeelClassName() {
        return lookAndFeelClassName;
    }

    /**
     * Sets the name of the Look and Feel class.
     * 
     * @param lnfClassName
     *            Full qualified Java LookAndFeel class name - System
     *            LookAndFeel is used when <code>null</code>.
     */
    public final void setLookAndFeelClassName(final String lnfClassName) {
        if (lnfClassName == null) {
            this.lookAndFeelClassName = UIManager.getSystemLookAndFeelClassName();
        } else {
            this.lookAndFeelClassName = lnfClassName;
        }
    }

    /**
     * Returns whether the program should terminate after executing.
     * 
     * @return If the program terminates with <code>System.exit(..)</code>
     *         <code>true</code> (DEFAULT) else <code>false</code>.
     */
    public final boolean isExitAfterExecute() {
        return exitAfterExecute;
    }

    /**
     * Sets whether the program should terminate after executing.
     * 
     * @param exitAfterExecute
     *            If the program should terminate with
     *            <code>System.exit(..)</code> <code>true</code> (DEFAULT) else
     *            <code>false</code>.
     */
    public final void setExitAfterExecute(final boolean exitAfterExecute) {
        this.exitAfterExecute = exitAfterExecute;
    }

    /**
     * Returns the URL of the directory where the source files are located.
     * 
     * @return URL - Always non-<code>null</code> if <code>check()</code> throws
     *         no exceptions.
     */
    public final URL getSrcUrl() {
        return srcUrl;
    }

    /**
     * Sets the URL of the directory where the source files are located.
     * 
     * @param srcUrl
     *            URL - Cannot be <code>null</code>
     */
    public final void setSrcUrl(final URL srcUrl) {
        if (srcUrl == null) {
            throw new IllegalArgumentException("The argument 'srcUrl' cannot be null!");
        }
        this.srcUrl = srcUrl;
    }

    /**
     * Sets the URL of the directory where the source files are located.
     * 
     * @param srcUrl
     *            URL - Cannot be <code>null</code>
     */
    public final void setSrcUrl(final String srcUrl) {
        if (srcUrl == null) {
            throw new IllegalArgumentException("The argument 'srcUrl' cannot be null!");
        }
        try {
            this.srcUrl = new URL(srcUrl);
        } catch (final MalformedURLException ex) {
            // Should be an IllegalArgumentException but 1.4 has no "String,
            // Throwable" constructor...
            throw new RuntimeException("The argument 'srcUrl' is not a valid URL [" + srcUrl
                    + "]!", ex);
        }
    }

    /**
     * Returns the local destination path where the files should be copied to.
     * 
     * @return Local path - Always non-<code>null</code> if <code>check()</code>
     *         throws no exceptions.
     */
    public final File getDestDir() {
        return destDir;
    }

    /**
     * Sets the local destination path where the files should be copied to.
     * 
     * @param destDir
     *            Directory - Cannot be <code>null</code>
     */
    public void setDestDir(final File destDir) {
        if (destDir == null) {
            throw new IllegalArgumentException("The argument 'destDir' cannot be null!");
        }
        this.destDir = destDir;
    }

    /**
     * Sets the local destination path where the files should be copied to.
     * 
     * @param destPath
     *            Path - Cannot be <code>null</code>
     */
    public final void setDestPath(final String destPath) {
        if (destPath == null) {
            throw new IllegalArgumentException("The argument 'destPath' cannot be null!");
        }
        this.destDir = new File(destPath);
    }

    /**
     * Returns the unique id filename for the application.
     * 
     * @return Name of the application ID file - Always non-<code>null</code> if
     *         <code>check()</code> throws no exceptions.
     */
    public final String getIdFilename() {
        return idFilename;
    }

    /**
     * Sets the unique id filename for the application.
     * 
     * @param id
     *            Unique ID - Must be a valid filename on the target system - A
     *            <code>null</code> value is not allowed!
     */
    public final void setIdFilename(final String id) {
        if (id == null) {
            throw new IllegalArgumentException("The argument 'id' cannot be null!");
        }
        this.idFilename = id;
    }

    /**
     * Determines if the user should be prompted for the destination directory.
     * 
     * @return If "destDir" should be used without asking the user
     *         <code>true</code> else <code>false</code>.
     */
    public final boolean isSilentInstall() {
        return silentInstall;
    }

    /**
     * Determines if the user should be prompted for the destination directory.
     * 
     * @param silentInstall
     *            If "destDir" should be used without asking the user
     *            <code>true</code> else <code>false</code>.
     */
    public final void setSilentInstall(final boolean silentInstall) {
        this.silentInstall = silentInstall;
    }

    /**
     * Determines if the user should be asked if an update is available.
     * 
     * @return If updates should be executed without asking the user
     *         <code>true</code> else <code>false</code>.
     */
    public final boolean isSilentUpdate() {
        return silentUpdate;
    }

    /**
     * Determines if the user should be asked if an update is available.
     * 
     * @param silentUpdate
     *            If updates should be executed without asking the user
     *            <code>true</code> else <code>false</code>.
     */
    public final void setSilentUpdate(final boolean silentUpdate) {
        this.silentUpdate = silentUpdate;
    }

    /**
     * Determines if this is the first installation.
     * 
     * @return If this is the first installation <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean isFirstInstallation() {
        return firstInstallation;
    }

    /**
     * Determines if this is the first installation.
     * 
     * @param firstInstallation
     *            If this is the first installation <code>true</code> else
     *            <code>false</code>.
     */
    public final void setFirstInstallation(final boolean firstInstallation) {
        this.firstInstallation = firstInstallation;
    }

    /**
     * Returns the locale to use.
     * 
     * @return Locale - Always non-<code>null</code>.
     */
    public final Locale getLocale() {
        return locale;
    }

    /**
     * Sets the locale to use.
     * 
     * @param locale
     *            Locale to set - A <code>null</code> value resets the property
     *            to <code>Locale.getDefault()</code>.
     */
    public final void setLocale(final Locale locale) {
        if (locale == null) {
            this.locale = Locale.getDefault();
        } else {
            this.locale = locale;
        }
    }

    /**
     * Sets the locale to use as a String.
     * 
     * @param locale
     *            Locale "lang" or "lang,country" or "lang,country,variant" - A
     *            <code>null</code> value resets the property to
     *            <code>Locale.getDefault()</code>.
     */
    public final void setLocale(final String locale) {
        if (locale == null) {
            this.locale = Locale.getDefault();
        } else {
            final StringTokenizer tok = new StringTokenizer(locale, ",");
            final int count = tok.countTokens();
            final String language;
            final String country;
            final String variant;
            if (count == 1) {
                language = tok.nextToken();
                country = "";
                variant = "";
            } else if (count == 2) {
                language = tok.nextToken();
                country = tok.nextToken();
                variant = "";
            } else if (count == 3) {
                language = tok.nextToken();
                country = tok.nextToken();
                variant = tok.nextToken();
            } else {
                throw new IllegalArgumentException("The argument '" + locale
                        + "' is not valid!");
            }
            this.locale = new Locale(language, country, variant);
        }
    }

    /**
     * Returns the file containing the localized installation messages.
     * 
     * @return File without path in the same directory as the JNLP file - If
     *         <code>null</code> the internal default messages will be used.
     */
    public final String getMsgFile() {
        return msgFile;
    }

    /**
     * Sets the file containing the localized installation messages.
     * 
     * @param msgFile
     *            File without path in the same directory as the JNLP file - A
     *            <code>null</code> value will result in usage of the internal
     *            default messages.
     */
    public final void setMsgFile(final String msgFile) {
        if (msgFile == null) {
            this.msgFile = null;
        } else {
            this.msgFile = msgFile.trim();
        }
    }

    /**
     * Checks if the configuration is valid.
     * 
     * @throws InvalidConfigException
     *             The configuration is not valid.
     */
    public final void check() throws InvalidConfigException {
        if (destDir == null) {
            throw new InvalidConfigException("The 'destDir' is null!");
        }
        if (srcUrl == null) {
            throw new InvalidConfigException("The 'srcUrl' is null!");
        }
        if (idFilename == null) {
            throw new InvalidConfigException("The 'idFilename' is null!");
        }
        if ((lazyLoading)
                && ((configFileUrlStr == null) || (configFileUrlStr.trim().length() == 0))) {
            throw new InvalidConfigException("The 'configFileUrlStr' is null or empty!");
        }
    }

    /**
     * Returns the message properties needed for the installer.
     * 
     * @return Localized messages.
     */
    public final Kickstart4JMessages getMessages() {
        if (messages == null) {
            try {
                messages = new Kickstart4JMessages(Utils4J.loadProperties(getSrcUrl(),
                        getMsgFile()));
            } catch (final RuntimeException ex) {
                // Load default messages
                messages = new Kickstart4JMessages(locale);
            }
        }

        return messages;
    }

    /**
     * Returns if lazy loading is active.
     * 
     * @return If lazy loading is enabled <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean isLazyLoading() {
        return lazyLoading;
    }

    /**
     * Sets the information if lazy loading is active.
     * 
     * @param lazyLoading
     *            To enable lazy loading <code>true</code> else
     *            <code>false</code> (disable lazy loading)).
     */
    public final void setLazyLoading(final boolean lazyLoading) {
        this.lazyLoading = lazyLoading;
    }

    /**
     * Adds a user defined option.
     * 
     * @param key
     *            Option key.
     * @param value
     *            Option value.
     */
    public final void addUserDefinedOption(final String key, final String value) {
        userDefinedOptions.put(key, value);
    }

    /**
     * Returns a user defined option value.
     * 
     * @param key
     *            Option key.
     * 
     * @return Option value.
     * 
     * @throws OptionNotFoundException
     *             The user defined option key is unknown.
     */
    public final String getUserDefinedOption(final String key) throws OptionNotFoundException {
        return (String) userDefinedOptions.get(key);
    }

    /**
     * Returns the user defined options as map.
     * 
     * @return Key/value <code>String</code> pairs.
     */
    public final Map getUserDefinedOptions() {
        return userDefinedOptions;
    }

    /**
     * Returns the command line arguments.
     * 
     * @return Arguments for the Java executable.
     */
    public final String getJavaArgs() {
        return javaArgs;
    }

    /**
     * Sets the command line arguments.
     * 
     * @param javaArgs
     *            Arguments for the Java executable.
     */
    public final void setJavaArgs(final String javaArgs) {
        this.javaArgs = javaArgs;
    }

    /**
     * Returns the Java executable.
     * 
     * @return Java executable.
     */
    public final String getJavaExe() {
        return javaExe;
    }

    /**
     * Sets the Java executable.
     * 
     * @param javaExe
     *            Java executable.
     */
    public final void setJavaExe(final String javaExe) {
        this.javaExe = javaExe;
    }

    /**
     * Returns a short description of the application.
     * 
     * @return Description.
     */
    public final String getDescription() {
        return description;
    }

    /**
     * Sets the short description of the application.
     * 
     * @param description
     *            Description.
     */
    public final void setDescription(final String description) {
        this.description = description;
    }

    /**
     * Returns the title of the application.
     * 
     * @return Title.
     */
    public final String getTitle() {
        return title;
    }

    /**
     * Sets the title of the application.
     * 
     * @param title
     *            Title.
     */
    public final void setTitle(final String title) {
        this.title = title;
    }

    /**
     * Returns the vendor information.
     * 
     * @return Vendor.
     */
    public final String getVendor() {
        return vendor;
    }

    /**
     * Sets the vendor information.
     * 
     * @param vendor
     *            Vendor.
     */
    public final void setVendor(final String vendor) {
        this.vendor = vendor;
    }

    /**
     * Returns the list of known source files.
     * 
     * @return Source file list.
     */
    public final List getSrcFiles() {
        return srcFiles;
    }

    /**
     * Returns the URL of the config file (only used when lazyLoading=true).
     * 
     * @return URL or <code>null</code>.
     */
    public final String getConfigFileUrlStr() {
        return configFileUrlStr;
    }

    /**
     * Sets the URL of the config file (only used when lazyLoading=true).
     * 
     * @param configFileUrlStr
     *            URL or <code>null</code>.
     */
    public final void setConfigFileUrlStr(final String configFileUrlStr) {
        this.configFileUrlStr = configFileUrlStr;
    }

    /**
     * Returns the target application version.
     * 
     * @return Version or <code>null</code> if not set.
     */
    public final String getVersion() {
        return version;
    }

    /**
     * Sets the target application version.
     * 
     * @param version
     *            Version or <code>null</code>.
     */
    public final void setVersion(final String version) {
        this.version = version;
    }

    /**
     * Finds a src file based on a local file reference.
     * 
     * @param file
     *            File to find a source file information for - Cannot be
     *            <code>null</code> and must be inside <code>destDir</code>.
     * 
     * @return Source file - Always non-<code>null</code>.
     * 
     * @throws SrcFileNotFoundException
     *             The corresponding source file was not found.
     */
    public final SrcFile findSrcFile(final File file) throws SrcFileNotFoundException {
        try {
            final String pathAndFilename = file.getCanonicalPath();
            for (int i = 0; i < srcFiles.size(); i++) {
                final SrcFile srcFile = (SrcFile) srcFiles.get(i);
                final String destPathAndFilename = srcFile
                        .getCanonicalPathAndFilename(destDir);
                if (pathAndFilename.equals(destPathAndFilename)) {
                    return srcFile;
                }
            }
            throw new SrcFileNotFoundException(Utils4J.getRelativePath(destDir, file
                    .getParentFile()), file.getName());
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Set a property by it's name. If the key is not known the key/value is
     * added using the <code>addUserDefinedOption</code> method.
     * 
     * @param key
     *            Name of the property.
     * @param value
     *            Value of the property.
     */
    public final void put(final String key, final String value) {
        if (key.equals("title")) {
            setTitle(value);
        } else if (key.equals("vendor")) {
            setVendor(value);
        } else if (key.equals("description")) {
            setDescription(value);
        } else if (key.equals("exitAfterExecute")) {
            setExitAfterExecute(toBoolean(value));
        } else if (key.equals("destPath")) {
            setDestPath(value);
        } else if (key.equals("idFilename")) {
            setIdFilename(value);
        } else if (key.equals("silentInstall")) {
            setSilentInstall(toBoolean(value));
        } else if (key.equals("silentUpdate")) {
            setSilentUpdate(toBoolean(value));
        } else if (key.equals("locale")) {
            setLocale(value);
        } else if (key.equals("lazyLoading")) {
            setLazyLoading(toBoolean(value));
        } else if (key.equals("javaExe")) {
            setJavaExe(value);
        } else if (key.equals("javaArgs")) {
            setJavaArgs(value);
        } else if (key.equals("srcUrl")) {
            setSrcUrl(value);
        } else if (key.equals("version")) {
            setVersion(value);
        } else {
            addUserDefinedOption(key, value);
        }
    }

    private boolean toBoolean(final String str) {
        if (str == null) {
            return false;
        }
        final String trimmed = str.trim();
        if (trimmed.length() == 0) {
            return false;
        }
        return Boolean.valueOf(trimmed).booleanValue();
    }

    /**
     * Returns the configuration as XML.
     * 
     * @return XML configuration.
     */
    public final String toXML() {
        final StringBuffer sb = new StringBuffer();
        sb.append("<application>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <version>" + version + "</version>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <title>" + title + "</title>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <vendor>" + vendor + "</vendor>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <description>" + description + "</description>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <exitAfterExecute>" + exitAfterExecute + "</exitAfterExecute>"
                + IOUtils.LINE_SEPARATOR);
        sb.append("  <srcUrl>" + srcUrl + "</srcUrl>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <destPath>" + destDir + "</destPath>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <idFilename>" + idFilename + "</idFilename>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <silentInstall>" + silentInstall + "</silentInstall>"
                + IOUtils.LINE_SEPARATOR);
        sb.append("  <silentUpdate>" + silentUpdate + "</silentUpdate>"
                + IOUtils.LINE_SEPARATOR);
        sb.append("  <locale>" + locale + "</locale>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <lazyLoading>" + lazyLoading + "</lazyLoading>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <javaExe>" + javaExe + "</javaExe>" + IOUtils.LINE_SEPARATOR);
        sb.append("  <javaArgs>" + javaArgs + "</javaArgs>" + IOUtils.LINE_SEPARATOR);
        for (int i = 0; i < srcFiles.size(); i++) {
            final SrcFile srcFile = (SrcFile) srcFiles.get(i);
            sb.append("  " + srcFile.toXML() + IOUtils.LINE_SEPARATOR);
        }
        sb.append("</application>" + IOUtils.LINE_SEPARATOR);
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     */
    public final String toString() {
        return toXML();
    }

}
