/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.File;
import java.net.URL;

import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.progress.FileCopyProgressListener;

/**
 * Supports loading files defined in a update set into a local directory.
 */
public final class SrcFileLoader implements FileLoader {

    /** Configuration with known source files. */
    private final Kickstart4JConfig config;

    /**
     * Constructor with URL.
     * 
     * @param configFileURL
     *            URL of the configuration file.
     * 
     * @throws InvalidConfigException
     *             Error reading the file.
     */
    public SrcFileLoader(final String configFileURL) throws InvalidConfigException {
        super();
        Utils4J.checkNotNull("configFileURL", configFileURL);
        this.config = Kickstart4JConfigParser.create(configFileURL);
    }

    /**
     * Constructor with configuration.
     * 
     * @param config
     *            Configuration.
     */
    public SrcFileLoader(final Kickstart4JConfig config) {
        super();
        Utils4J.checkNotNull("config", config);
        this.config = config;
    }

    /**
     * Returns the configuration.
     * 
     * @return Configuration used for loading the files.
     */
    public final Kickstart4JConfig getConfig() {
        return config;
    }

    /**
     * {@inheritDoc}
     */
    public void loadFile(final File file) throws LoadingFileException {
        loadFile(file, null);
    }

    /**
     * {@inheritDoc}
     */
    public void loadFile(final File file, final FileCopyProgressListener listener)
            throws LoadingFileException {

        Utils4J.checkNotNull("file", file);
        Utils4J.checkValidDir(file.getParentFile());

        try {
            final SrcFile srcFile = config.findSrcFile(file);
            final URL srcFileUrl = srcFile.getUrl(config.getSrcUrl());
            final File destFile = srcFile.getDestFile(config.getDestDir());

            if (destFile.exists()) {
                final String destHash = Utils4J.createHash(destFile);
                if (!srcFile.getMd5Hash().equals(destHash)) {
                    // Changed file
                    Utils.copyURLToFile(listener, srcFileUrl, destFile, 1, srcFile.getSizeAsInt());
                }
            } else {
                // New file
                Utils.copyURLToFile(listener, srcFileUrl, destFile, 1, srcFile.getSizeAsInt());
            }
        } catch (final SrcFileNotFoundException ex) {
            throw new LoadingFileException("Unknown file '" + file + "'!", ex);
        } catch (final RuntimeException ex) {
            throw new LoadingFileException("Error loading file '" + file + "'!", ex);
        }

    }

}
