/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.File;
import java.net.URL;

import org.fuin.utils4j.Utils4J;

/**
 * A file at the source directory. This class is immutable.
 */
public final class SrcFile extends AbstractFile {

    /** Token used to mark archives to be unzipped at the target location. */
    public static final String UNZIP_TOKEN = "UNZIP";

    /** Size of the file. */
    private final long size;

    /** ZIP file that needs to be decompressed at the target location. */
    private final boolean unzip;

    /** Load file always (no matter if lazy loading is enabled or not). */
    private final boolean loadAlways;

    /** JAR file to be added to the classpath. */
    private final boolean addToClasspath;

    /**
     * Constructor with base directory and file.
     * 
     * @param baseDir
     *            Base directory for calculating the relative path.
     * @param file
     *            File inside the base directory.
     * @param unzip
     *            If the file is an archive and should be decompressed at the
     *            target location <code>true</code> else <code>false</code>.
     * @param loadAlways
     *            Load file always (no matter if lazy loading is enabled or
     *            not).
     * @param addToClasspath
     *            JAR file to be added to the classpath.
     */
    public SrcFile(final File baseDir, final File file, final boolean unzip,
            final boolean loadAlways, final boolean addToClasspath) {
        super(baseDir, file);
        this.size = file.length();
        this.unzip = unzip;
        this.loadAlways = loadAlways;
        this.addToClasspath = addToClasspath;
    }

    /**
     * Constructor with all arguments.
     * 
     * @param path
     *            Path relative to the base directory.
     * @param filename
     *            Filename without path.
     * @param md5Hash
     *            MD5 hash code of the file.
     * @param size
     *            Size of the file.
     * @param unzip
     *            If the file is an archive and should be decompressed at the
     *            target location <code>true</code> else <code>false</code>.
     * @param loadAlways
     *            Load file always (no matter if lazy loading is enabled or
     *            not).
     * @param addToClasspath
     *            JAR file to be added to the classpath.
     */
    public SrcFile(final String path, final String filename, final String md5Hash, final long size,
            final boolean unzip, final boolean loadAlways, final boolean addToClasspath) {
        super(path, filename, md5Hash);
        this.size = size;
        this.unzip = unzip;
        this.loadAlways = loadAlways;
        this.addToClasspath = addToClasspath;
    }

    /**
     * Returns the size of the file.
     * 
     * @return File size.
     */
    public final long getSize() {
        return size;
    }

    /**
     * Returns the size as integer value. If the size is greater than
     * <code>Integer.MAX_VALUE</code> is returned.
     * 
     * @return Size.
     */
    public final int getSizeAsInt() {
        if (size > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        } else {
            return (int) size;
        }
    }

    /**
     * Returns if the file is an atchive and should be decompressed at the
     * target location.
     * 
     * @return Unzip the file at the destination directory <code>true</code>
     *         else <code>false</code>
     */
    public final boolean isUnzip() {
        return unzip;
    }

    /**
     * Returns an URL based on the given source directory URL.
     * 
     * @param srcUrl
     *            Source directory URL.
     * 
     * @return File URL inside the source directory.
     */
    public final URL getUrl(final URL srcUrl) {
        return Utils4J.createUrl(srcUrl, getPath().replace(File.separatorChar, '/'), getFilename());
    }

    /**
     * Returns if this is a JAR file to be added to the classpath.
     * 
     * @return If added to the classpath <code>true</code> else
     *         <code>false</code>.
     */
    public final boolean isAddToClasspath() {
        return addToClasspath;
    }

    /**
     * Returns if this file sould be loaded always (no matter if lazy loading is
     * enabled or not).
     * 
     * @return If loaded always <code>true</code> else <code>false</code>.
     */
    public final boolean isLoadAlways() {
        return loadAlways;
    }

    /**
     * {@inheritDoc}
     */
    public final String toXML() {
        return "<file path=\"" + getSlashPath() + "\" file=\"" + getFilename() + "\" hash=\""
                + getMd5Hash() + "\" size=\"" + size + "\" unzip=\"" + unzip + "\" loadAlways=\""
                + loadAlways + "\" addToClasspath=\"" + addToClasspath + "\"/>";
    }

    /**
     * {@inheritDoc}
     */
    public final String toString() {
        return toXML();
    }    
    
}
