/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.kickstart4j;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.launcher.CommandLauncher;
import org.apache.commons.exec.launcher.CommandLauncherFactory;
import org.fuin.utils4j.Cancelable;
import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.common.Utils4Swing;
import org.fuin.utils4swing.dialogs.CanceledException;
import org.fuin.utils4swing.dialogs.DirectorySelector;
import org.fuin.utils4swing.progress.FileCopyProgressListener;
import org.fuin.utils4swing.progress.FileCopyProgressMonitor;
import org.fuin.utils4swing.threadsafe.ThreadSafeJOptionPane;

/**
 * Main application.
 */
public final class Kickstart4J {

    private static final String INCOMPLETE_FILE = ".incomplete";

    private static final String PROGRAM_DIRECTORY_KEY = "program-directory";

    /** Configuration used for the application. */
    private final Kickstart4JConfig config;

    /**
     * Constructor with configuration.
     * 
     * @param config
     *            Configuration to use.
     */
    public Kickstart4J(final Kickstart4JConfig config) {
        super();
        if (config == null) {
            throw new IllegalArgumentException("The argument 'config' cannot be null!");
        }
        this.config = config;
    }

    /**
     * Executes the installer/updater.
     * 
     * @throws CanceledException
     *             The user canceled the installation.
     * @throws InvalidConfigException
     *             The configuration is invalid.
     */
    public final void execute() throws CanceledException, InvalidConfigException {
        Locale.setDefault(config.getLocale());
        final File destDir = getDestDir();
        // Check config after destination directory is finally set
        config.check();

        // Start the update
        final UpdateSet updateSet = new UpdateSet(config.getSrcFiles(), destDir, config
                .isLazyLoading());
        if (updateSet.isUpdateNecessary()
                && (config.isSilentUpdate() || config.isFirstInstallation() || isAnswerYes(config
                        .getMessages().getUpdateAvailable()))) {
            execute(updateSet);
            final File installationIncompleteFile = new File(destDir, INCOMPLETE_FILE);
            if (installationIncompleteFile.exists()) {
                installationIncompleteFile.delete();
            }
        }

        // Run the target application
        final Map varMap = new HashMap();
        varMap.put("classpath", createClasspath(updateSet));
        varMap.put("configFileUrl", config.getConfigFileUrlStr());
        varMap.putAll(config.getUserDefinedOptions());

        final CommandLine commandLine = new CommandLine(new File(config.getDestDir(), config
                .getJavaExe()));
        commandLine.setSubstitutionMap(varMap);
        commandLine.addArguments(config.getJavaArgs(), false);
        logStart(destDir, commandLine.toString());
        startExternal(destDir, commandLine);

    }

    private String createClasspath(final UpdateSet updateSet) {
        final List cpJarFiles = updateSet.getClasspathJarFiles();
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < cpJarFiles.size(); i++) {
            final SrcFile srcFile = (SrcFile) cpJarFiles.get(i);
            if (i > 0) {
                sb.append(File.pathSeparatorChar);
            }
            sb.append("\"");
            sb.append(srcFile.getRelativeSlashPathAndFilename());
            sb.append("\"");
        }
        return sb.toString();
    }

    private File getIdFile(final Kickstart4JConfig config) {
        return new File(Utils4J.getUserHomeDir(), config.getIdFilename());
    }

    private void execute(final UpdateSet updateSet) throws CanceledException {
        executeCopy(updateSet);
        executeDecompress(updateSet);
    }

    private void executeCopy(final UpdateSet updateSet) throws CanceledException {

        final int max = updateSet.getNewFiles().size() + updateSet.getChangedFiles().size();
        if (max > 0) {

            final Cancelable cancelable = new Cancelable() {

                private volatile boolean canceled = false;

                public void cancel() {
                    canceled = true;
                }

                public boolean isCanceled() {
                    return canceled;
                }
            };

            final FileCopyProgressMonitor monitor = new FileCopyProgressMonitor(cancelable, config
                    .getMessages().getProgressMonitorTitle(), config.getMessages()
                    .getProgressMonitorTransferText(), config.getMessages()
                    .getProgressMonitorSrcLabelText(), config.getMessages()
                    .getProgressMonitorDestLabelText(), max);

            int count = 0;

            monitor.open();
            try {

                // New files
                if (cancelable.isCanceled()) {
                    throw new CanceledException();
                }
                count = copyFiles(updateSet, cancelable, monitor, updateSet.getNewFiles(), count);

                // Changed files
                if (cancelable.isCanceled()) {
                    throw new CanceledException();
                }
                count = copyFiles(updateSet, cancelable, monitor, updateSet.getChangedFiles(),
                        count);

                // No longer existent (deleted) files
                if (cancelable.isCanceled()) {
                    throw new CanceledException();
                }
                for (int i = 0; i < updateSet.getDeletedFiles().size(); i++) {
                    if (cancelable.isCanceled()) {
                        break;
                    }
                    count = count + 1;
                    final String file = (String) updateSet.getDeletedFiles().get(i);
                    final File destFile = new File(updateSet.getDestDir(), file);
                    monitor.updateFile("", destFile.toString(), count, 0);
                }

            } finally {
                monitor.close();
            }

        }
    }

    private void executeDecompress(final UpdateSet updateSet) throws CanceledException {

        final List compressedFiles = updateSet.getDecompressFiles();
        final int max = compressedFiles.size();
        if (max > 0) {

            final Cancelable cancelable = new Cancelable() {

                private volatile boolean canceled = false;

                public void cancel() {
                    canceled = true;
                }

                public boolean isCanceled() {
                    return canceled;
                }
            };

            final FileCopyProgressMonitor monitor = new FileCopyProgressMonitor(cancelable, config
                    .getMessages().getProgressMonitorTitle(), config.getMessages()
                    .getProgressMonitorDecompressText(), config.getMessages()
                    .getProgressMonitorSrcLabelText(), config.getMessages()
                    .getProgressMonitorDestLabelText(), max);

            monitor.open();
            try {

                for (int i = 0; i < max; i++) {
                    final SrcFile file = (SrcFile) compressedFiles.get(i);
                    final File compressedFile = file.getDestFile(updateSet.getDestDir());
                    Utils.unzip(monitor, compressedFile, (i + 1), updateSet.getDestDir(),
                            cancelable);
                }

            } finally {
                monitor.close();
            }

        }
    }

    private int copyFiles(final UpdateSet updateSet, final Cancelable cancelable,
            final FileCopyProgressListener listener, final List files, final int total) {

        int count = total;

        for (int i = 0; i < files.size(); i++) {
            if (cancelable.isCanceled()) {
                break;
            }
            count = count + 1;
            final SrcFile file = (SrcFile) files.get(i);
            final URL srcFileUrl = file.getUrl(config.getSrcUrl());
            final File destFile = file.getDestFile(updateSet.getDestDir());
            Utils.copyURLToFile(listener, srcFileUrl, destFile, count, file.getSizeAsInt());
        }

        return count;

    }

    private void logStart(final File dir, final String commandLine) {
        try {
            final File file = new File(dir, "start.log");
            final FileWriter writer = new FileWriter(file);
            try {
                writer.write(commandLine);
            } finally {
                writer.close();
            }
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    private void startExternal(final File destDir, final CommandLine commandLine) {
        final CommandLauncher launcher = CommandLauncherFactory.createVMLauncher();
        try {
            launcher.exec(commandLine, null, destDir);
            // TODO Get errors from new process if it does not start
            // TODO Display something like "Starting application..." for X
            // seconds
            System.exit(0);
        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    private File getDestDir() throws CanceledException {
        final File installationIncompleteFile;
        final File dir;
        final File idFile = getIdFile(config);
        if (idFile.exists()) {
            // Update
            final Properties props = Utils4J.loadProperties(idFile);
            final String dirStr = props.getProperty(PROGRAM_DIRECTORY_KEY);
            if (dirStr == null) {
                throw new IllegalStateException("The property '" + PROGRAM_DIRECTORY_KEY
                        + "' was not found inside '" + idFile + "'!");
            }
            dir = new File(dirStr);
            Utils4J.checkValidDir(dir);
            installationIncompleteFile = new File(dir, INCOMPLETE_FILE);
        } else {
            // First installation or file removed
            final String dirStr;
            final Properties props = new Properties();
            if (config.isSilentInstall()) {
                dirStr = config.getDestDir().toString();
            } else {
                // Ask User for destination directory
                dirStr = DirectorySelector.selectDirectory(
                        config.getMessages().getSelectDestinationDirectory(),
                        config.getDestDir().toString()).getDirectory();
            }
            props.setProperty(PROGRAM_DIRECTORY_KEY, dirStr);
            Utils4J.saveProperties(idFile, props,
                    "# --- DO NOT EDIT --- Generated by Kickstart4J ---");
            dir = new File(dirStr);
            installationIncompleteFile = new File(dir, INCOMPLETE_FILE);
            if (!dir.exists()) {
                try {
                    dir.mkdirs();
                    installationIncompleteFile.createNewFile();
                } catch (final IOException ex) {
                    throw new RuntimeException("Cannot create file '" + installationIncompleteFile
                            + "'!");
                }
            }
        }
        config.setFirstInstallation(installationIncompleteFile.exists());
        return dir;
    }

    private static void initLookAndFeel(final String lookAndFeelClassName) {
        try {
            SwingUtilities.invokeAndWait(new Runnable() {
                public void run() {
                    Utils4Swing.initLookAndFeel(lookAndFeelClassName);
                }
            });
        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    private static boolean isAnswerYes(final String message) {
        final int result = ThreadSafeJOptionPane.showConfirmDialog(null, message, "TITLE",
                JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
        return result == JOptionPane.YES_OPTION;
    }

    private static void showError(final String message) {
        ThreadSafeJOptionPane.showMessageDialog(null, message, "Error", JOptionPane.ERROR_MESSAGE);
    }

    private static void showMessage(final String message) {
        ThreadSafeJOptionPane.showMessageDialog(null, message, "Hint",
                JOptionPane.INFORMATION_MESSAGE);
    }

    private static void displayCmdLine() {
        final String msg = "java Kickstart4J <CONFIG-FILE-URL>";
        System.out.println(msg);
        showError(msg);
    }

    private static void displayException(final Exception ex) {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        ex.printStackTrace(new PrintStream(out));
        final String msg = out.toString();
        System.err.println(msg);
        showError(msg);
    }

    private static File createLocalConfigFile() {
        try {
            return File.createTempFile("application-", ".xml");
        } catch (final IOException ex) {
            throw new RuntimeException("Cannot create local config file!", ex);
        }
    }

    private static String toUrlString(final File localConfigFile) {
        try {
            return localConfigFile.toURI().toURL().toString();
        } catch (final MalformedURLException ex) {
            throw new RuntimeException("Cannot create local config file URL!", ex);
        }
    }

    /**
     * Main method used to start the installer/updater. If you want to start it
     * from another Java application you can simply use
     * <code>new Kickstart4J(kickstart4JConfig).execute()</code> instead of
     * calling this method.
     * 
     * @param args
     *            The only argument is the URL of the configuration XML file.
     */
    public static void main(final String[] args) {

        if ((args == null) || (args.length != 1)) {
            displayCmdLine();
            return;
        }

        final String remoteConfigFileUrlStr = args[0].trim();
        try {
            // Copy the config to a temporary file
            final File localConfigFile = createLocalConfigFile();
            final String localConfigFileUrlStr = toUrlString(localConfigFile);
            Utils.copyURLToFile(remoteConfigFileUrlStr, localConfigFile);

            // Load the config from local file and start update
            final Kickstart4JConfig config = new Kickstart4JConfig();
            try {
                Kickstart4JConfigParser.parse(config, localConfigFileUrlStr);
                config.setConfigFileUrlStr(localConfigFileUrlStr);

                initLookAndFeel(config.getLookAndFeelClassName());
                (new Kickstart4J(config)).execute();

            } catch (final CanceledException ex) {
                showMessage(config.getMessages().getOperationCanceled());
                System.exit(0);
            } catch (final InvalidConfigException ex) {
                showMessage(ex.getMessage());
                System.exit(0);
            }

        } catch (final RuntimeException ex) {
            displayException(ex);
            System.exit(0);
        }
    }

}
