/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.ControllerPair;
import org.fuin.jmsmvc4swing.base.ControllerResultListener;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.jms.ControllerReceiver;
import org.fuin.jmsmvc4swing.jms.ControllerSender;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Model for generating classes based on a <code>Controller</code> interface.
 * 
 * @param <T>
 *            A controller interface.
 */
public final class ControllerModel<T extends Controller> {

    private final ByteCodeGenerator generator;

    private final SgClassPool pool;

    private final String basePackage;

    private final SgClass ctrlInterface;

    private final List<SgClass> classes;

    /**
     * Constructor with controller interface the generated classes are based on.
     * The destination package for the generated classes is the same as the
     * interface.
     * 
     * @param controllerInterface
     *            Interface.
     */
    @SuppressWarnings("unchecked")
    public ControllerModel(final Class controllerInterface) {
        this(controllerInterface.getPackage().getName(), controllerInterface, null);
    }

    /**
     * Constructor with controller interface the generated classes are based on
     * and the destination package for the generated classes.
     * 
     * @param basePackage
     *            Package all generated classes are inside.
     * @param controllerInterface
     *            Interface.
     */
    @SuppressWarnings("unchecked")
    public ControllerModel(final String basePackage, final Class controllerInterface) {
        this(basePackage, controllerInterface, null);
    }

    /**
     * Constructor with controller interface, destination package for the
     * generated classes and byte code generator.
     * 
     * @param basePackage
     *            Package all generated classes are inside.
     * @param controllerInterface
     *            Interface.
     * @param generator
     *            Byte code generator - If NULL a default byte code generator
     *            instance will be created and used.
     */
    @SuppressWarnings("unchecked")
    public ControllerModel(final String basePackage, final Class controllerInterface,
            final ByteCodeGenerator generator) {
        super();
        if (basePackage == null) {
            throw new IllegalArgumentException("The argument 'basePackage' cannot be null!");
        }
        this.basePackage = basePackage;

        if (controllerInterface == null) {
            throw new IllegalArgumentException(
                    "The argument 'controllerInterface' cannot be null!");
        }
        this.classes = new ArrayList<SgClass>();

        this.pool = new SgClassPool();

        this.ctrlInterface = SgClass.create(pool, controllerInterface);

        if (generator == null) {
            this.generator = new ByteCodeGenerator();
        } else {
            this.generator = generator;
        }

        check();
        parse();
    }

    private void check() {
        if (!ctrlInterface.isInterface()) {
            throw new IllegalArgumentException(
                    "Only interfaces are allowed for argument 'controllerInterface'! ["
                            + ctrlInterface.getName() + " is not an interface!]");
        }

        final List<SgMethod> methods = ctrlInterface.getMethods();
        if (methods.size() == 0) {
            throw new IllegalArgumentException("There are no methods within the interface! ["
                    + ctrlInterface.getName() + "]");
        }

        final List<SgClass> listenerInterfaces = getControllerResultListener(ctrlInterface
                .getClasses());
        if (methods.size() != listenerInterfaces.size()) {
            throw new IllegalArgumentException("There number of interfaces ("
                    + listenerInterfaces.size() + ") is not equal to the number of methods ("
                    + methods.size() + ")!");
        }

        for (int i = 0; i < methods.size(); i++) {
            final SgMethod method = methods.get(i);
            final SgArgument lastArgument = method.getLastArgument();
            if (lastArgument == null) {
                throw new IllegalArgumentException("The method '" + method.getName()
                        + "' has no arguments but at least a "
                        + "'ControllerResultListener' is mandantory!");
            }
            final String listenerName = ctrlInterface.getName() + "$"
                    + SgUtils.firstCharUpper(method.getName()) + "Listener";
            if (!lastArgument.getType().getName().equals(listenerName)) {
                throw new IllegalArgumentException("The last argument of method "
                        + method.getName() + " is type '" + lastArgument.getType().getName()
                        + "' and not '" + listenerName + "' as expected!");
            }
            final SgClass listenerClass = ctrlInterface.findClassByName(listenerName);
            if (listenerClass == null) {
                throw new IllegalArgumentException("Interface " + listenerName + " not found!");
            }

        }
    }

    /**
     * Returns a list of the generated model classes.
     * 
     * @return List of classes.
     */
    public final List<SgClass> getModelClasses() {
        return classes;
    }

    /**
     * Returns the controller interface the model is based on.
     * 
     * @return Controller interface.
     */
    public final SgClass getControllerInterface() {
        return ctrlInterface;
    }

    /**
     * Name of the interface converted into a package name. Inserts an
     * underscore before every upper case character and returns an all lower
     * case string. If the first character is upper case an underscore will not
     * be inserted.
     * 
     * @return Package for the generated classes.
     */
    public final String getPackageName() {
        return ctrlInterface.getSimpleNameAsPackage();
    }

    /**
     * Returns the name of the interface.
     * 
     * @return Name without package.
     */
    public final String getSimpleName() {
        return ctrlInterface.getSimpleName();
    }

    private void parse() {

        final List<SgMethod> intfMethods = ctrlInterface.getMethods();

        // Create classes not depending to other generated classes
        for (int i = 0; i < intfMethods.size(); i++) {

            final SgMethod intfMethod = intfMethods.get(i);
            final ArgumentsGenerator<T> argsGen = new ArgumentsGenerator<T>(pool, this,
                    basePackage, intfMethod);
            classes.add(argsGen.createModelClass());

            // Result classes
            final String listenerName = ctrlInterface.getName() + "$"
                    + SgUtils.firstCharUpper(intfMethod.getName()) + "Listener";
            final SgClass listenerClass = ctrlInterface.findClassByName(listenerName);
            final List<SgMethod> listenerMethods = listenerClass.getMethods();
            for (int j = 0; j < listenerMethods.size(); j++) {
                final SgMethod listenerMethod = listenerMethods.get(j);
                final ResultGenerator<T> resultGen = new ResultGenerator<T>(pool, this,
                        basePackage, listenerMethod);
                classes.add(resultGen.createModelClass());
            }
        }

        // Create classes with first degree dependencies
        for (int i = 0; i < intfMethods.size(); i++) {
            final SgMethod intfMethod = intfMethods.get(i);

            final MethodResultSenderGenerator<T> mrsGen = new MethodResultSenderGenerator<T>(
                    pool, this, basePackage, intfMethod);
            classes.add(mrsGen.createModelClass());

            final MethodResultReceiverGenerator<T> mrrGen = new MethodResultReceiverGenerator<T>(
                    pool, this, basePackage, intfMethod);
            classes.add(mrrGen.createModelClass());
        }

        // Create classes with second degree dependencies
        for (int i = 0; i < intfMethods.size(); i++) {
            final SgMethod intfMethod = intfMethods.get(i);

            final MethodHandlerGenerator<T> mhGen = new MethodHandlerGenerator<T>(pool, this,
                    basePackage, intfMethod);
            classes.add(mhGen.createModelClass());

            final MethodCallSenderGenerator<T> mcsGen = new MethodCallSenderGenerator<T>(pool,
                    this, basePackage, intfMethod);

            classes.add(mcsGen.createModelClass());

        }

        // Create the rest...
        final ControllerReceiverGenerator<T> crGen = new ControllerReceiverGenerator<T>(pool,
                this, basePackage);
        classes.add(crGen.createModelClass());

        final ControllerSenderGenerator<T> csGen = new ControllerSenderGenerator<T>(pool,
                this, basePackage);
        classes.add(csGen.createModelClass());

    }

    /**
     * Creates all classes "on-the-fly" with Javassist and then returns the
     * necessary instances to work with. Alternatively you can use
     * <code>createSource(..)</code> to generate source instead of bytecode
     * 
     * @param env
     *            Environment to use for the instances.
     * @param topicName
     *            Topic name to use.
     * @param ctrl
     *            Real controller implementation doing the work.
     * 
     * @return Two new created instances based on the "on-the-fly" created
     *         classes.
     */
    @SuppressWarnings("unchecked")
    public final ControllerPair<T> createOnTheFly(final JmsJndiEnvironment env,
            final String topicName, final Controller ctrl) {

        Class senderClass = null;
        Class receiverClass = null;

        // Create classes
        for (int i = 0; i < classes.size(); i++) {
            final SgClass modelClass = classes.get(i);

            final Class clasz = generator.createClass(modelClass);
            if (modelClass.getSuperClass() != null) {
                if (modelClass.getSuperClass().getName().equals(
                        ControllerReceiver.class.getName())) {
                    receiverClass = clasz;
                } else if (modelClass.getSuperClass().getName().equals(
                        ControllerSender.class.getName())) {
                    senderClass = clasz;
                }
            }
        }

        if (senderClass == null) {
            throw new IllegalStateException("No 'sender' found!");
        }
        if (receiverClass == null) {
            throw new IllegalStateException("No 'receiver' found!");
        }

        final ControllerSender sender = (ControllerSender) generator.createInstance(
                senderClass, new Class[] { JmsJndiEnvironment.class, String.class },
                new Object[] { env, topicName });

        final ControllerReceiver receiver = (ControllerReceiver) generator.createInstance(
                receiverClass, new Class[] { JmsJndiEnvironment.class, String.class,
                        Controller.class }, new Object[] { env, topicName, ctrl });

        return new ControllerPair((T) sender, receiver);

    }

    /**
     * Creates the source classes for the model. Alternatively you can use
     * <code>createOnTheFly(..)</code> to generate bytecode instead of source.
     * 
     * @param srcDir
     *            Destination directory.
     * 
     * @throws IOException
     *             Error creating the classes.
     */
    public void createSource(final File srcDir) throws IOException {

        for (int i = 0; i < classes.size(); i++) {
            final SgClass modelClass = classes.get(i);
            final String filename = modelClass.getNameAsSrcFilename();
            final File srcFile = new File(srcDir, filename);
            srcFile.getParentFile().getCanonicalFile().mkdirs();
            final Writer writer = new BufferedWriter(new FileWriter(srcFile));
            try {
                writer.write(modelClass.toString());
            } finally {
                writer.close();
            }
        }

    }

    private static List<SgClass> getControllerResultListener(final List<SgClass> classes) {

        final List<SgClass> listener = new ArrayList<SgClass>();
        for (int i = 0; i < classes.size(); i++) {
            final List<SgClass> interfaces = classes.get(i).getInterfaces();
            for (int j = 0; j < interfaces.size(); j++) {
                if (interfaces.get(j).getName().equals(
                        ControllerResultListener.class.getName())) {
                    listener.add(classes.get(i));
                }
            }
        }
        return listener;
    }

    /**
     * Returns the byte code generator the model uses.
     * 
     * @return Byte code generator.
     */
    public final ByteCodeGenerator getGenerator() {
        return generator;
    }

    /**
     * Returns the class pool the model uses.
     * 
     * @return Class pool.
     */
    public final SgClassPool getPool() {
        return pool;
    }

}
