/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates a <code>ControllerReceiver</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 * @param <T>
 *            Type of the class the generator will create.
 */
public abstract class AbstractGenerator<I extends Controller, T> {

    private final SgClassPool pool;

    private final ControllerModel<I> model;

    private final String basePackage;

    private final String subPackage;

    private final String simpleDestClass;

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param subPackage
     *            Package within base package.
     * @param simpleDestClass
     *            Name of the destination class (without package).
     */
    public AbstractGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage, final String subPackage, final String simpleDestClass) {
        super();

        if (pool == null) {
            throw new IllegalArgumentException("The argument 'pool' cannot be null!");
        }
        this.pool = pool;

        if (model == null) {
            throw new IllegalArgumentException("The argument 'model' cannot be null!");
        }
        this.model = model;

        if (basePackage == null) {
            throw new IllegalArgumentException("The argument 'basePackage' cannot be null!");
        }
        this.basePackage = basePackage;

        if (subPackage == null) {
            throw new IllegalArgumentException("The argument 'subPackage' cannot be null!");
        }
        this.subPackage = subPackage;

        if (simpleDestClass == null) {
            throw new IllegalArgumentException(
                    "The argument 'simpleDestClass' cannot be null!");
        }
        this.simpleDestClass = simpleDestClass;
    }

    /**
     * Returns the controller model.
     * 
     * @return Controller model.
     */
    public final ControllerModel<I> getModel() {
        return model;
    }

    /**
     * Returns the base package.
     * 
     * @return Package all generated classes will be inside.
     */
    protected final String getBasePackage() {
        return basePackage;
    }

    /**
     * Package within the base package.
     * 
     * @return Package the generated class will be in.
     */
    protected final String getSubPackage() {
        return subPackage;
    }

    /**
     * Simple name (without package) of the class that will be generated.
     * 
     * @return Simple name of the destination class.
     */
    protected final String getSimpleDestClass() {
        return simpleDestClass;
    }

    /**
     * Full package name.
     * 
     * @return Package name (base + sub).
     */
    protected final String getDestPackage() {
        return SgUtils.concatPackages(basePackage, subPackage);
    }

    /**
     * Full name of the class.
     * 
     * @return Name including base package, sub package and simple name.
     */
    protected final String getDestClass() {
        return SgUtils.concatPackages(getDestPackage(), simpleDestClass);
    }

    /**
     * Returns the class pool.
     * 
     * @return Class pool.
     */
    protected final SgClassPool getPool() {
        return pool;
    }

}
