/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import java.io.Serializable;

import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.base.ProgressInfo;

/**
 * Sends results from the controller with a JMS message to the subscribers.
 */
public abstract class MethodResultSender {

    private final JmsJndiEnvironment env;

    private final String topicName;

    private final String methodName;

    private final long id;

    /**
     * Constructor.
     * 
     * @param env
     *            Environment to use.
     * @param topicName
     *            Name of the topic (defined by the controller) to send to.
     * @param methodName
     *            Name of the method that was called (used for creating the
     *            message properties).
     * @param id
     *            Unique id of the method call.
     */
    public MethodResultSender(final JmsJndiEnvironment env, final String topicName,
            final String methodName, final long id) {
        super();
        this.env = env;
        this.topicName = topicName;
        this.methodName = methodName;
        this.id = id;
    }

    /**
     * Publish a message.
     * 
     * @param type
     *            Type of the message.
     * @param serializable
     *            Data.
     */
    private void publishIntern(final MessageType type, final Serializable serializable) {
        final Publisher publisher = new Publisher(env);
        publisher.open(topicName);
        try {
            publisher.publish(methodName, serializable, id, type);
        } finally {
            publisher.close();
        }
    }

    /**
     * Publish a RESULT message.
     * 
     * @param data
     *            Result data.
     */
    protected final void publishResult(final Data data) {
        publishIntern(ResponseMessageType.RESULT, data);
    }

    /**
     * Publish a STARTED message.
     */
    public final void started() {
        publishIntern(ResponseMessageType.STARTED, null);
    }

    /**
     * Publish a PROGESS event.
     * 
     * @param progressInfo
     *            Progress information.
     */
    public final void progress(final ProgressInfo progressInfo) {
        publishIntern(ResponseMessageType.PROGRESS, progressInfo);
    }

    /**
     * Publish a SUCCEDED message.
     */
    public final void succeded() {
        publishIntern(ResponseMessageType.SUCCEDED, null);
    }

    /**
     * Publish a FAILED message.
     * 
     * @param ex
     *            Exception.
     */
    public final void failed(final Exception ex) {
        publishIntern(ResponseMessageType.FAILED, ex);
    }

}
