/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import java.util.HashMap;
import java.util.Map;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.IControllerReceiver;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;

/**
 * Receives a JMS message for the controller and selects the appropriate
 * handler.
 */
public class ControllerReceiver implements MessageListener, IControllerReceiver {

    /** Topic the listener subscribes to. */
    private final String topicName;

    /** Controller to use. */
    private final Controller ctrl;

    /** Map of known methods and their handlers. */
    private final Map<String, MethodHandler> handlerMap;

    /** Handles the JMS subscriptions. */
    private final Subscriber subscriber;

    /**
     * Constructor with all necessary data.
     * 
     * @param env
     *            Context.
     * @param topicName
     *            Topic the listener subscribes to.
     * @param ctrl
     *            Controller to use.
     */
    public ControllerReceiver(final JmsJndiEnvironment env, final String topicName,
            final Controller ctrl) {
        super();
        this.ctrl = ctrl;
        this.handlerMap = new HashMap<String, MethodHandler>();
        this.subscriber = new Subscriber(env);
        this.topicName = topicName;
    }

    /**
     * Returns the topic for this controller.
     * 
     * @return Topic.
     */
    public final String getTopicName() {
        return topicName;
    }

    /**
     * {@inheritDoc}
     */
    public final void start() {
        subscriber.open(topicName, null, null, RequestMessageType.START, this);
    }

    /**
     * Select the appropriate handler based on a method name.
     * 
     * @param methodName
     *            Name of the method to be called.
     * @param args
     *            Arguments for the method.
     * @param id
     *            Unique method call ID.
     */
    private void handle(final String methodName, final Data args, final long id) {
        final MethodHandler handler = handlerMap.get(methodName);
        if (handler == null) {
            // TODO Handle this correctly!
            System.out.println("ERROR - Unknown method! [" + Constants.METHOD + "="
                    + methodName + ", args=" + args + ", " + Constants.ID + "=" + id + "]");
        } else {
            handler.handle(ctrl, getTopicName(), args, id);
        }
    }

    /**
     * {@inheritDoc}
     */
    public final void onMessage(final Message message) {
        try {
            if (!(message instanceof ObjectMessage)) {
                // TODO Handle this correctly!
                System.out.println("ERROR - No object message! [message=" + message + "]");
            }
            final ObjectMessage om = (ObjectMessage) message;
            final Object obj = om.getObject();
            if (!(obj instanceof Data)) {
                // TODO Handle this correctly!
                System.out.println("ERROR - No argument! [object=" + obj + "]");
            }
            handle(om.getStringProperty(Constants.METHOD), (Data) obj, om
                    .getLongProperty(Constants.ID));
        } catch (final JMSException ex) {
            // TODO Handle this correctly!
            ex.printStackTrace();
        }
    }

    /**
     * {@inheritDoc}
     */
    public final boolean isStopped() {
        return subscriber.isClosed();
    }

    /**
     * {@inheritDoc}
     */
    public final void stop() {
        subscriber.close();
    }

    /**
     * Adds a new handler for a controller method.
     * 
     * @param handler
     *            Handles a single JMS controller message.
     */
    public final void addRequestHandler(final MethodHandler handler) {
        handlerMap.put(handler.getMethodName(), handler);

    }

}
