/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import java.util.Arrays;

/**
 * Request message send to the controller.
 */
public final class RequestMessageType implements MessageType {

    /** Call a method. */
    public static final RequestMessageType START = new RequestMessageType("START");

    /** List of all known instances. */
    private static final RequestMessageType[] INSTANCES = new RequestMessageType[] { START };

    /** Name of the type. */
    private final String name;

    /**
     * Private constructor.
     * 
     * @param name
     *            Name of the type.
     */
    private RequestMessageType(final String name) {
        super();
        this.name = name;
    }

    /**
     * {@inheritDoc}
     */
    public final String getName() {
        return name;
    }

    /**
     * Returns if a give name is valid.
     * 
     * @param name
     *            Type name.
     * 
     * @return If the name is valid TRUE else FALSE.
     */
    public static boolean isValid(final String name) {
        for (int i = 0; i < INSTANCES.length; i++) {
            if (INSTANCES[i].getName().equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the instance for a name.
     * 
     * @param name
     *            Name to get the (singleton) instance for.
     * 
     * @return Instance.
     */
    public static RequestMessageType forName(final String name) {
        for (int i = 0; i < INSTANCES.length; i++) {
            if (INSTANCES[i].getName().equals(name)) {
                return INSTANCES[i];
            }
        }
        throw new IllegalArgumentException("Unknown name '" + name + "'!");
    }

    /**
     * Returns a list of all kown instances.
     * 
     * @return Copy of the internal instances list.
     */
    public static RequestMessageType[] getAll() {
        return Arrays.copyOf(INSTANCES, INSTANCES.length);
    }

}
