/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;

import org.fuin.jmsmvc4swing.base.ControllerResultListener;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.base.ProgressInfo;

/**
 * Base class for receivers of a method result.
 */
public abstract class MethodResultReceiver implements MessageListener {

    private final Subscriber subscriber;

    private final String topicName;

    private final String methodName;

    private final long id;

    private final ControllerResultListener listener;

    /**
     * Constructor.
     * 
     * @param env
     *            Environment to use.
     * @param topicName
     *            Name of the topic (defined by the controller) to send to.
     * @param methodName
     *            Name of the method that was called (used for creating the
     *            message properties).
     * @param id
     *            Unique id of the method call.
     * @param listener
     *            Listener to be informed if messages arrive.
     * 
     */
    public MethodResultReceiver(final JmsJndiEnvironment env, final String topicName,
            final String methodName, final long id, final ControllerResultListener listener) {
        super();
        this.topicName = topicName;
        this.subscriber = new Subscriber(env);
        this.methodName = methodName;
        this.id = id;
        this.listener = listener;
    }

    /**
     * Start subscription to the predefined topic and with the appropriate
     * selector.
     */
    public final void subscribe() {
        subscriber.open(topicName, methodName, id, ResponseMessageType.getAll(), this);
    }

    /**
     * {@inheritDoc}
     */
    public final void onMessage(final Message message) {
        if (!(message instanceof ObjectMessage)) {
            // TODO Handle this correctly!
            System.out.println("ERROR - No object message! [message=" + message + "]");
        }
        try {
            final ObjectMessage msg = (ObjectMessage) message;
            final ResponseMessageType type = ResponseMessageType.forName(msg
                    .getStringProperty(Constants.TYPE));
            if (type.equals(ResponseMessageType.STARTED)) {
                listener.started();
            } else if (type.equals(ResponseMessageType.PROGRESS)) {
                listener.progress((ProgressInfo) msg.getObject());
            } else if (type.equals(ResponseMessageType.RESULT)) {
                handleResult((Data) msg.getObject());
            } else if (type.equals(ResponseMessageType.SUCCEDED)) {
                subscriber.close();
                listener.succeded();
            } else if (type.equals(ResponseMessageType.FAILED)) {
                subscriber.close();
                listener.failed((Exception) msg.getObject());
            }
        } catch (final JMSException ex) {
            // TODO Handle this correctly!
            ex.printStackTrace();
        }
    }

    /**
     * Checks if the subscription was already cancelled or finished.
     * 
     * @return If the subscription is already closed TRUE else FALSE.
     */
    public final boolean isCancelled() {
        return subscriber.isClosed();
    }

    /**
     * Cancels the subscription.
     */
    public final void cancel() {
        System.out.println("CANCEL");
        subscriber.close();
    }

    /**
     * Call the concrete message result receiver.
     * 
     * @param result
     *            Data received.
     */
    protected abstract void handleResult(final Data result);

}
