/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.base;

import javax.jms.JMSException;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.naming.Context;
import javax.naming.NamingException;

/**
 * Default environment for using JMS and JNDI.
 */
public final class JmsJndiEnvironmentImpl implements JmsJndiEnvironment {

    /** Prefix for all topic names. */
    public static final String CONTEXT = "dynamicTopics";

    /** Naming context. */
    private final Context context;

    /** JMS provider connection. */
    private TopicConnection connection;

    /** Key used to lookup a topic connection factory with JNDI. */
    private final String topicConnectionFactoryKey;

    /**
     * Constructor with all necessary data.
     * 
     * @param context
     *            Naming context.
     * @param topicConnectionFactoryKey
     *            Key used to lookup a topic connection factory with JNDI.
     */
    public JmsJndiEnvironmentImpl(final Context context, final String topicConnectionFactoryKey) {
        super();
        this.context = context;
        this.topicConnectionFactoryKey = topicConnectionFactoryKey;
    }

    /**
     * {@inheritDoc}
     */
    public final void open() {
        try {
            final TopicConnectionFactory factory = (TopicConnectionFactory) context
                    .lookup(topicConnectionFactoryKey);
            connection = factory.createTopicConnection();
            connection.start();
        } catch (final JMSException ex) {
            throw new RuntimeException(ex);
        } catch (final NamingException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * {@inheritDoc}
     */
    public final boolean isOpen() {
        return connection != null;
    }

    /**
     * {@inheritDoc}
     */
    public final void close() {
        if (isOpen()) {
            try {
                connection.close();
            } catch (final JMSException e) {
                e.printStackTrace();
            }
            connection = null;
        }
    }

    /**
     * {@inheritDoc}
     */
    public final Topic getTopic(final String name) {
        try {
            return (Topic) context.lookup(CONTEXT + "/" + name);
        } catch (final NamingException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * {@inheritDoc}
     */
    public final TopicConnection getConnection() {
        return connection;
    }

    /**
     * {@inheritDoc}
     */
    public final Context getContext() {
        return context;
    }

}
