/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import java.util.Arrays;

/**
 * Response message send by the controller.
 */
public final class ResponseMessageType implements MessageType {

    /** Controller started working on a method. */
    public static final ResponseMessageType STARTED = new ResponseMessageType("STARTED");

    /** Controller signaling progress. */
    public static final ResponseMessageType PROGRESS = new ResponseMessageType("PROGRESS");

    /** Controller sends result. */
    public static final ResponseMessageType RESULT = new ResponseMessageType("RESULT");

    /** Controller successfully finished working on a method. */
    public static final ResponseMessageType SUCCEDED = new ResponseMessageType("SUCCEDED");

    /** Controller finished working on a method with an exception. */
    public static final ResponseMessageType FAILED = new ResponseMessageType("FAILED");

    /** List of all known instances. */
    private static final ResponseMessageType[] INSTANCES = new ResponseMessageType[] {
            STARTED, PROGRESS, RESULT, SUCCEDED, FAILED };

    /** Name of the type. */
    private final String name;

    /**
     * Private constructor.
     * 
     * @param name
     *            Name of the type.
     */
    private ResponseMessageType(final String name) {
        super();
        this.name = name;
    }

    /**
     * {@inheritDoc}
     */
    public final String getName() {
        return name;
    }

    /**
     * Returns if a give name is valid.
     * 
     * @param name
     *            Type name.
     * 
     * @return If the name is valid TRUE else FALSE.
     */
    public static boolean isValid(final String name) {
        for (int i = 0; i < INSTANCES.length; i++) {
            if (INSTANCES[i].getName().equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the instance for a name.
     * 
     * @param name
     *            Name to get the (singleton) instance for.
     * 
     * @return Instance.
     */
    public static ResponseMessageType forName(final String name) {
        for (int i = 0; i < INSTANCES.length; i++) {
            if (INSTANCES[i].getName().equals(name)) {
                return INSTANCES[i];
            }
        }
        throw new IllegalArgumentException("Unknown name '" + name + "'!");
    }

    /**
     * Returns a list of all kown instances.
     * 
     * @return Copy of the internal instances list.
     */
    public static ResponseMessageType[] getAll() {
        return Arrays.copyOf(INSTANCES, INSTANCES.length);
    }

}
