/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import java.io.Serializable;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;

import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;

/**
 * Publishes messages to a JMS topic.
 */
public final class Publisher {

    private final JmsJndiEnvironment env;

    private TopicSession session;

    private TopicPublisher publisher;

    /**
     * Constructor.
     * 
     * @param env
     *            Environment to use.
     */
    public Publisher(final JmsJndiEnvironment env) {
        super();
        this.env = env;
    }

    /**
     * Create a topic to publish to (Opens a session to the JMS provider).
     * 
     * @param topicName
     *            Name of the topic.
     */
    public final void open(final String topicName) {
        try {
            this.session = env.getConnection().createTopicSession(false,
                    Session.AUTO_ACKNOWLEDGE);
            final Topic topic = env.getTopic(topicName);
            this.publisher = session.createPublisher(topic);
        } catch (final JMSException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Returns if a JMS session is active.
     * 
     * @return If a session is in use TRUE else FALSE.
     */
    public final boolean isOpen() {
        return (session != null);
    }

    /**
     * Returns if no JMS session is active.
     * 
     * @return If no session is in use TRUE else FALSE.
     */
    public final boolean isClosed() {
        return (session == null);
    }

    /**
     * Close the session to the JMS provider.
     */
    public final void close() {
        if (isOpen()) {
            try {
                session.close();
                session = null;
                publisher = null;
            } catch (final JMSException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    /**
     * Publishes a message to the predefined topic.
     * 
     * @param methodName
     *            Name of the method for creating the selector.
     * @param data
     *            Data for the message.
     * @param id
     *            Id for creating the selector.
     * @param type
     *            Types for creating the selector.
     */
    public final void publish(final String methodName, final Serializable data, final long id,
            final MessageType type) {
        try {
            final ObjectMessage message = session.createObjectMessage();
            message.setStringProperty(Constants.METHOD, methodName);
            message.setStringProperty(Constants.ID, "" + id);
            message.setStringProperty(Constants.TYPE, type.getName());
            if (data != null) {
                message.setObject(data);
            }
            System.out.println("PUBLISH [" + publisher.getTopic().getTopicName() + "] "
                    + Utils.getMessageSelector(methodName, id, type) + " / DATA=" + data);
            publisher.publish(message);
        } catch (final JMSException ex) {
            throw new RuntimeException(ex);
        }
    }

}
