/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;

/**
 * Base class for units sending a method call to a controller.
 */
public abstract class MethodCallSender {

    private final JmsJndiEnvironment env;

    private final String topicName;

    private final boolean determinate;

    /**
     * Constructor.
     * 
     * @param env
     *            Environment to use.
     * @param topicName
     *            Name of the topic (defined by the controller) to send to.
     * @param determinate
     *            Information if the progress of the method is determinate.
     */
    public MethodCallSender(final JmsJndiEnvironment env, final String topicName,
            final boolean determinate) {
        this.env = env;
        this.topicName = topicName;
        this.determinate = determinate;
    }

    /**
     * Returns the environment.
     * 
     * @return Environment.
     */
    protected final JmsJndiEnvironment getEnvironment() {
        return env;
    }

    /**
     * Returns the topic name.
     * 
     * @return Name of the topic (defined by the controller) to send to.
     */
    protected final String getTopicName() {
        return topicName;
    }

    /**
     * Send a JMS message to the controller and register a subscriber to receive
     * the results.
     * 
     * @param methodName
     *            Name of the method for creating the selector.
     * @param arguments
     *            Arguments for the method call.
     * @param id
     *            Id for creating the selector.
     * @param resultListener
     *            Subscriber for the results.
     */
    protected final void publish(final String methodName, final Data arguments, final long id,
            final MethodResultReceiver resultListener) {
        final Publisher publisher = new Publisher(env);
        publisher.open(topicName);
        try {
            resultListener.subscribe();
            publisher.publish(methodName, arguments, id, RequestMessageType.START);
        } finally {
            publisher.close();
        }
    }

    /**
     * Returns if the method will send progress informations with min/max/value.
     * 
     * @return If determinate TRUE else FALSE.
     */
    public final boolean isDeterminate() {
        return determinate;
    }

}
